/*    WIPort ALL [platforms] 2
 *
 *    Copyright (c) 1994--1998 Alexandru Dan Corlan
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU Library General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    In short, you can use and modify this program. You can
 *    distribute the program with or without your modifications if you want,
 *    but you must not require licence fees and must provide the
 *    full source code with the distribution. If you make an application program
 *    which just calls functions from this library you may distribute that
 *    application program without source code, but also as an linkable object file, 
 *    and must distribute also this library with source code.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Library General Public License for more details.
 *
 *    You should have received a copy of the GNU Library General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*--

  \FileTitle{ % format: <short title> --- <long title>
  WIPALL2 --- platform independent part of WIPORT-2
  }
  \CopyrightNotice{ % format: Copyright \copyright <year(s)> <who>
  Copyright \copyright 1994 Alexandru Dan CORLAN
  }
  \FileAuthor{ % the full name of the author
  Alexandru Dan CORLAN
  }
  \AuthorAffiliation{ % Intitution of the author when writing the file
  Electrocardiognosis SRL, 28 Argentina str, 71206, Bucharest 1, ROMANIA
  }
  \RevisionHistory{ % write freely the revision history here
  Created 23-rd of October 1994, from WIPORT1.C
  }
  \vskip 1in

  \chapter{WIPALL2 implementation}

  \PackagePurpose{  % be short
  }
  \FunctionInterraction{
  % The order in which functions are expected to be called
  % The way to use the package in general
  }
  \Exceptions{
  }
  \GlobalEffects{
  % Global effects on exported or external variables
  }
  \Preconditions{
  % Conditions to be met when the package starts to be used
  }
  \Performance{
  % How it is, not why and what to do
  }

  \begin{code}
  --*/

#include <stdio.h>
#include <malloc.h>
#include <sys/types.h>
#include "wipall2.h"

extern short wi_widget_background;
extern short wi_widget_darkborder;
extern short wi_widget_lightborder;
void _tp_format( struct _tp_ext * this, short width );

short gwerrno;

/* Exception management 
 */
typedef char * (*WIPORT_EXCEPTION_HANDLER)();

static WIPORT_EXCEPTION_HANDLER wiport_exception_handler;

void wiport_set_exception_handler( WIPORT_EXCEPTION_HANDLER  excep ) {
  wiport_exception_handler = excep; 
}

void wraise( char * e ) {
  if( wiport_exception_handler != NULL )  {}
    (*wiport_exception_handler)( e );
}

/*--
  \end{code}

  \section{GWIN registry}

  \ImplementationNotes{
  % How implementation of the whole package is done: algorithms---if special,
  % implementation decisions, discarded implementation alternatives
  The maximum number of gwins is made local because
  it is really no use to change it. WIPORT is NOT intented for
  applications with as many as 17 gwins.
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  NO ERRORS are returned by the registration function; if more complex
  application come into existence this will have to be taken
  into account.

  \verb|wi_get_crt_gwin| should be implemented as a macro; it is 
  called by most of the functions, isn't it.
  }

  \begin{code}
  --*/

#define MAX_NOF_GWINS 16  

static GWIN *gwins[MAX_NOF_GWINS];
static short gwinfill = 0; /* fill pointer in the above */
static short crt_gwin;

void _register_gwin(GWIN *g)

{
  if(gwinfill>=MAX_NOF_GWINS)
    return ;
  crt_gwin = gwinfill;
  gwins[gwinfill++] = g;
  return ;
}

short wi_set_crt_gwin(short g)

{
  if(g<0 || g>=gwinfill)
    return 0; /* error */
  crt_gwin = g;
  return 1;
}

gwin wi_get_crt_gwin()

{
  return crt_gwin;
}

GWIN *wi_crt_gwin_ptr()

{
return gwins[crt_gwin];
}


/*--
  \end{code}

  \section{Application wide functions}

  \ImplementationNotes{
  % How implementation of the whole package is done: algorithms---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/


static char wi_app_name[64] = "WIPORT-2.0";

void wi_set_app_name(char *an)

{
  strncpy(wi_app_name, an, 63);
}

char *wi_get_app_name()

{
  return wi_app_name;
}


/*--
  \end{code}

  \section{Internals of gwin and dwin management}

  \ImplementationNotes{
  % How implementation of the whole package is done: algorithms---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/


short _add_grel_page(DWIN *dw) /* [FUN] */ 
     
{
  short i;
  
  /* add a new grel page */
  if(dw->gpfill >= NOGPPERDWIN)
    {
      gwerrno = OVF_GPPERDWIN;
      return 0;
    }
  dw->gp[dw->gpfill] = malloc(GPSIZE);
  if(dw->gp[dw->gpfill]==NULL)
    {
      gwerrno = NO_MEMORY_FOR_GP;
      return 0;
    }
  for(i=0; i<NOGPERPAGE; i++)
    dw->gp[dw->gpfill][i].ext = NULL;
  
  dw->gpfill++;
  return 1;
}

DWIN *dwalloc(short x0, short y0, short xl, short yl) { /* [FUN] */ 
  DWIN *r;
  short i;
  
  r = (DWIN *)malloc(sizeof(DWIN));
  if(r==NULL)
    {
      gwerrno = NO_MEMORY_FOR_DWIN;
      return r;
    }
  r->x0 = x0; 
  r->y0 = y0;
  r->xl = xl;
  r->yl = yl;
  r->cr_draw_x = r->cr_draw_y = 0;
  r->cr_col = 8;
  r->cr_style = 0xffff;
  r->cr_width = 1;
  r->cr_cap = 0;
  r->spotshotkey = r->horshotkey = r->vershotkey = r->rectshotkey = 0;
  for(i=0; i<NOGPPERDWIN; i++)
    r->gp[i] = NULL;
  r->gpfill = 0;
  r->gppfill = 0;
  _add_grel_page(r);
  return r;
}

/* Data structures and functions that permit iterations over the grels.
 */

ITERATOR * init_fwd_grel_iterator(GWIN * gw, short d, ITERATOR * this) { /* [FUN] */ 
  this -> dw = gw -> dw[d];
  this -> current_page = this -> index_in_page = 0;
  return this;
}

ITERATOR * init_bwd_grel_iterator(GWIN * gw, short d, ITERATOR * this) { /* [FUN] */ 
  this -> dw = gw -> dw[d];
  this -> current_page = this -> dw -> gpfill - 1;
  this -> index_in_page = this -> dw -> gppfill - 1;
}

ITERATOR * iterator_forward(ITERATOR * this) { /* [FUN] */ 
  if (++ this -> index_in_page >= NOGPERPAGE) {
    this -> current_page ++;
    this -> index_in_page = 0;
  }
  return this;
}

ITERATOR * iterator_backward(ITERATOR * this) { /* [FUN] */ 
  if (this -> index_in_page--)
    return this;
  this -> index_in_page = NOGPERPAGE - 1;
  if (this -> current_page--)
    return this;
  return NULL;
}

int check_upper_bound_grel(ITERATOR * this) { /* [FUN] */ 
  if (((this -> current_page < (this -> dw -> gpfill -1))
       && (this -> index_in_page < NOGPERPAGE))
      || ((this -> current_page == (this -> dw -> gpfill - 1))
          && (this -> index_in_page < this -> dw -> gppfill)))
    return 1;
  return 0;
}

int check_lower_bound_grel(ITERATOR * this) { /* [FUN] */ 
  return ((this -> current_page >= 0) && (this -> index_in_page >= 0));
}

struct _grel * access_grel(ITERATOR * this) { /* [FUN] */ 
  return ((this -> dw -> gp)[this -> current_page] + this -> index_in_page);
}

/* Function to access the last grel of a dwin */
struct _grel * last_grel (DWIN * dw) { /* [FUN] */ 
  if ((dw -> gpfill == 1) && (dw -> gppfill = 0))
    return NULL;
  if (dw -> gppfill)
    return (dw -> gp[dw -> gpfill - 1] + dw -> gppfill - 1);
  else
    return (dw -> gp[dw -> gpfill - 2] + NOGPERPAGE - 1);
}

struct _grel * ndx_to_grel( DWIN * dw, int ndx ) {
  return ((dw -> gp)[ ndx / NOGPERPAGE ] + ndx % NOGPERPAGE);
}

/* Follow functions that are replics of the highest-level
 * functions but only affect the dwin representation
 */

void _gpclear(grelpage g) { /* [FUN] */ 
  short i;
  
  for(i=0; i<NOGPERPAGE; i++)
    if(g[i].ext!=NULL)
      {
        free(g[i].ext);
        g[i].ext = NULL;
      }
  return ;
}

short _inc_gpfill(DWIN *dw) { /* [FUN] */ 
  dw->gppfill ++;
  if(dw->gppfill>=NOGPERPAGE)
    {
      if(_add_grel_page(dw)==0) {
	wraise( "EGrelManagement" );
      }
      dw -> gppfill = 0; 
    }
  return 1;
}


void _dwin_clear(DWIN *dw) /* [FUN] */ 
     /* all grels in the dwin are deleted, and all pages
        disallocated, with the exception of the first page ---
        which is cleared but remains for further output
        */
     
{
  short i;
  
  for(i=0; i<dw->gpfill; i++) {
    _gpclear(dw->gp[i]);
    if(i>0) {
      free(dw->gp[i]);
      dw -> gp[i] = NULL;
    }
  }
  dw->gpfill=1;
  dw->gppfill=0;
  dw->spotshotkey = dw->horshotkey = dw->vershotkey =
    dw->rectshotkey = 0;
  return ;
}

static inline struct _grel* 
_new_grel( DWIN * dw ) {
  return dw->gp[dw->gpfill-1] + dw->gppfill;
}

static inline short
_bump_grel( DWIN * dw ) {
  short r = NOGPERPAGE * (dw->gpfill - 1) + dw->gppfill;

  if(!_inc_gpfill(dw))
    {
      return -1;
    }
  return r;
}

/* Function: _init_grel
 * Purpose:  performs default initialization on the grels
 */
static inline void
_init_grel( struct _grel * g, int type, pixed x0, pixed y0
	   , pixed xl, pixed yl, short key, int col ) {
  g->gtype=type;
  g->x0 = x0;
  g->x1 = x0+xl;
  g->y0 = y0;
  g->y1 = y0+yl;
  g->gkey = key;
  g->gcolor = col;
  g->ext=NULL;
  g->gstyle = g->gsize = g->gcap = 0;
}

int definthbutton(DWIN *dw, short key) {
  struct _grel *g = _new_grel( dw );
  _init_grel( g, HBUTTONGREL, 0, 0, 0, 0, 0, dw->cr_col );
  g->swvval = 0;
  g->gswv = NULL;

  return _bump_grel(dw);
}


int defintbutton(DWIN *dw, short x0, short y0, short xl, short yl, char *text, short key) /* [FUN] */ 
     
{
  struct _grel *g = _new_grel(dw);
  short r = _bump_grel(dw);  if( r==-1) return r;
  _init_grel( g, BUTTONGREL, x0, y0, xl, yl, key, dw->cr_col );
  g->swvval = 0;
  g->gswv = NULL;

  g->ext = malloc(strlen(text)+1);
  if(g->ext==NULL)
    {
      gwerrno = NO_MEMORY_FOR_BUTTEXT;
      return 0;
    }
  strcpy(g->ext, text);
  g->extfill = g->extsize = strlen(text);

  return r;
}

int defintswitch(DWIN *dw, short x0, short y0, short xl, short yl, char *text, short *swv, short key) /* [FUN] */ 
     
{
  struct _grel *g=_new_grel(dw);
  short r = _bump_grel(dw);

  if( r==-1 ) return r;
  _init_grel( g, SWITCHGREL, x0, y0, xl, yl, key, dw->cr_col );
  g->gkey = key;
  g->gswv = swv;
  g->swvval = *swv;   /* This is the internal value */
  g->ext = malloc(strlen(text)+1);
  if(g->ext==NULL)
    {
      gwerrno = NO_MEMORY_FOR_SWITEXT;
      return 0;
    }
  strcpy(g->ext, text);
  g->extfill = g->extsize = strlen(text);
  return r;
}

int defintaswitch(DWIN *dw, short x0, short y0, short xl, short yl, char *text, short *swv, short key) /* [FUN] */ 
     
{
  struct _grel *g = _new_grel(dw);
  short r = _bump_grel(dw); if( r==-1 ) return r;

  _init_grel( g, ASWITCHGREL, x0, y0, xl, yl, key, dw->cr_col );
  g->gswv = swv;
  g->swvval = *swv;   /* This is the internal value */

  g->ext = malloc(strlen(text)+1);
  if(g->ext==NULL)
    {
      gwerrno = NO_MEMORY_FOR_ALTSWITEXT;
      return 0;
    }
  strcpy(g->ext, text);
  g->extfill = g->extsize = strlen(text);
  return r;
}

int insert_grel_coord(struct _grel *g, short x, short y) { /* [FUN] */ 
  /* the next function inserts the coordinates of a new grel
   * in a MULTIGREL
   */
  if (g -> ext == NULL) {
    g -> ext = malloc(sizeof(short) * 8); /* a simple strategy */
    if (g -> ext == NULL) return 0;
    g -> extsize = 8;
    g -> extfill = 0;
  }
  if (g -> extfill == g -> extsize) return 0; /* a simple strategy */
  ((short*)g->ext)[ g->extsize++ ] = x;
  ((short*)g->ext)[ g->extsize++ ] = y;
  return 1;
}

#define pointgrel_is_like(g, color, style) ((g -> gcolor == color) && (g -> gstyle == style) ) /* [DEF] */ 

int defintpoint(DWIN *dw, short x0, short y0, short col) { /* [FUN] */ 
  struct _grel *g;

  short r;
/*
  g = last_grel(dw);
  if (g != NULL) {
    if ((g -> gtype == POINTGREL)
        && pointgrel_is_like(g,col,0)) {
      if (insert_grel_coord(g, x0, y0)) {
        g -> gtype == MULTIPOINTGREL;
        return 1;
      }
    } else if ((g -> gtype == MULTIPOINTGREL)
               && pointgrel_is_like(g,col,0)) {
      if (insert_grel_coord(g, x0, y0))
        return 1;
    }
  } */

  g = _new_grel( dw );
  r = _bump_grel( dw ); if( r==-1 ) return r;
  _init_grel( g, POINTGREL, x0, y0, 1, 1, 0 , dw->cr_col);
  return r;
}

#define linegrel_is_like(g, color, style, width, cap, x, y) (\
                                                             (g -> gcolor == color)\
                                                             && (g -> x1 == x)\
                                                             && (g -> y1 == y)\
                                                             && (g -> gstyle == style)\
                                                             && (g -> gsize == width)\
                                                             && (g -> gcap == cap)\
                                                             )

int defintline(DWIN *dw, short x0, short y0, short x1, short y1, short col, short width, short style, short cap) { /* [FUN] */ 
  struct _grel *g;
  short r;

/*  
  g = last_grel(dw);
  if (g != NULL) {
    if ((g -> gtype == LINEGREL)
        && linegrel_is_like(g,col,style,width,cap,x0,y0)) {
      if (insert_grel_coord(g, x0, y0)) {
        g -> gtype == MULTILINEGREL;
        return 1;
      }
    } else if ((g -> gtype == MULTILINEGREL)
               && linegrel_is_like(g,col,style,width,cap,x0,y0)) {
      if (insert_grel_coord(g, x0, y0))
        return 1;
    }
  } */
  g = _new_grel(dw);
  r = _bump_grel(dw); if( r==-1 ) return r;
  _init_grel( g, LINEGREL, x0, y0, x1, y1, 0 , dw->cr_col);
  g->gcolor = col;
  g->gstyle = style;
  g->gsize = width;
  g->gcap = cap;
  return r;
}

int defintrect( DWIN *dw, short x0, short y0, short x1, short y1, short col ) {
  struct _grel * g = _new_grel(dw);
  short r = _bump_grel(dw); if(r==-1) return r;
  
  _init_grel( g, RECTGREL, x0, y0, x1, y1, 0, dw->cr_col );
  return r;
}

int definttext(DWIN *dw, short x0, short y0, short col, char *text) /* [FUN] */ 
{
  struct _grel *g = _new_grel(dw);
  short r = _bump_grel(dw); if( r==-1 ) return r;

  _init_grel( g, TEXTGREL, x0, y0, col * wi_get_char_x_size(), wi_get_char_y_size(), 0, dw->cr_col);

  g->ext = malloc(strlen(text)+1);
  if(g->ext==NULL)
    {
      gwerrno = NO_MEMORY_FOR_TEXT;
      return 0;
    }
  strcpy(g->ext, text);
  g->extfill = g->extsize = strlen(text);

  return r;
}
  
int definttextpad(DWIN *dw, short x0, short y0, short nc, short nl, char * buffer, short size, short key) { /* [FUN] */ 
  struct _grel * g = _new_grel(dw);
  short r = _bump_grel(dw); if( r==-1 ) return r;

  _init_grel( g, TEXTPADGREL, x0, y0, nc * wi_get_char_x_size()
	     , nl * wi_get_char_y_size()+1, key, dw->cr_col );
  g -> gkey = key; 
  g -> gswv = (void *)buffer;
  g -> ext = malloc( sizeof(struct _tp_ext) );

  _tp_init( g->ext, buffer, size );
  _tp_format( g->ext, nc*wi_get_char_x_size() );
  
  return r;
}

short defintgrel(  dwin d, pixed x0, pixed y0
		   , pixed  width, pixed height ) {
  DWIN *dw=wi_crt_gwin_ptr()->dw[d];
  struct _grel * g = _new_grel(dw);
  short r = _bump_grel(dw); if( r==-1 ) return r;

  _init_grel( g, BITMAPGREL, x0, y0, width, height, 0, dw->cr_col );
  return r;
}


/*--
  \end{code}

  \section{Technologies}

  \ImplementationNotes{
  We have chosen to define default colors here. An implementation
  for a given platform might ignore these definition and redefine them,
  or may use the defaults as a starting point. The second approach
  is recommended in order to obtain uniformity of the default conditions
  on different platforms.
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/


static short nr_of_ccodes = 16;

/* The following dynamically allocated vectors 
 * contain nr_of_codes elements each.
 * Vector 0 is for screen, 1 for paper, 2 for slide.
 */

static short *bwtech[3] = { NULL, NULL, NULL };  
static shade *graystech[3] = { NULL, NULL, NULL };
static shade *ctech[3] = { NULL, NULL, NULL };
static shade *mtech[3] = { NULL, NULL, NULL };
static shade *ytech[3] = { NULL, NULL, NULL };
static shade *ktech[3] = { NULL, NULL, NULL };

short crt_screen_tech = COLOR_SCREEN;
short crt_hardcopy_tech = GRAYS_PAPER;

/*--
  \end{code}

  \funparagraph{\_allocate\_color\_codes}
  \ImplementationNotes{
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/


short _allocate_color_codes(short n)   /* the number of color codes */

{
  short i, j, p;

  nr_of_ccodes = n;
  
  for(i=0; i<3; i++) {
    bwtech[i] = (short *)malloc(n * sizeof(short));
    if(bwtech[i]==NULL)
      goto allocfailed;
    for(j=0; j<n; j++)
      if(j==0)
	bwtech[i][j] = 0;
      else
	bwtech[i][j] = 1;  /* all are black on white background */
  }

  for(i=0; i<3; i++) {
    graystech[i] = (shade *)malloc(n * sizeof(shade));
    if(graystech[i]==NULL)
      goto allocfailed;
    for(j=0; j<n; j++)
      if(j==0)
	graystech[i][j] = 0;
      else
	graystech[i][j] = (shade)(((j+1) * (long)1000) / n);
  }

  for(i=0; i<3; i++) {
    ctech[i] = (shade *)malloc(n * sizeof(shade));
    if(ctech[i]==NULL)
      goto allocfailed;
    mtech[i] = (shade *)malloc(n * sizeof(shade));
    if(mtech[i]==NULL)
      goto allocfailed;
    ytech[i] = (shade *)malloc(n * sizeof(shade));
    if(ytech[i]==NULL)
      goto allocfailed;
    ktech[i] = (shade *)malloc(n * sizeof(shade));
    if(ktech[i]==NULL)
      goto allocfailed;
    p = n/8;
    for(j=0; j<n; j++) {
      if(j & 0x0001)
	ytech[i][j] = 1000;
      else
	ytech[i][j] = 0;
      if(j & 0x0002)
	ctech[i][j] = 1000;
      else
	ctech[i][j] = 0;
      if(j & 0x0004)
	mtech[i][j] = 1000;
      else
	mtech[i][j] = 0;
      ktech[i][j] = 1000 - (1 + j/8) * (1000 / p);
    }
  }

  return 0;  /* done, no error */
  
 allocfailed:
  for(i=0; i<3; i++) {
    if(bwtech[i]!=NULL) {
      free(bwtech[i]);
      bwtech[i] = NULL;
    }
    if(graystech[i]!=NULL) {
      free(graystech[i]);
      graystech[i] = NULL;
    }
    if(ctech[i]!=NULL) {
      free(ctech[i]);
      ctech[i] = NULL;
    }
    if(mtech[i]!=NULL) {
      free(mtech[i]);
      mtech[i] = NULL;
    }
    if(ytech[i]!=NULL) {
      free(ytech[i]);
      ytech[i] = NULL;
    }
    if(ktech[i]!=NULL) {
      free(ktech[i]);
      ktech[i] = NULL;
    }
  }
  rraise(NO_MEMORY);
  return -1;
}



/*--
  \end{code}

  \funparagraph{\_use\_xx\_tech}
  \ImplementationNotes{
  These functions set the current technology. As they usually
  have also an effect on the screen (changing the colors), that
  effect is provided by the WIPORT-2 primitives which
  call these function.
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

short _use_screen_tech(technology t)

{
  if(t<COLOR_SCREEN || t>BW_SCREEN) {
    rraise(INVALID_TECHNOLOGY);
    return 0;
  }
  crt_screen_tech = t;
  return 1;
}

short _use_hardcopy_tech(technology t)

{
  if(t<COLOR_PAPER || t>BW_SLIDE) {
    rraise(INVALID_TECHNOLOGY);
    return 0;
  }
  crt_hardcopy_tech = t;
  return 1;
}


/*--
  \end{code}

  \funparagraph{\_get\_xx\_tech}  % The name of the function(s)
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/


technology wi_get_screen_tech()

{
  return crt_screen_tech;
}

technology wi_get_hardcopy_tech()

{
  return crt_hardcopy_tech;
}


/*--
  \end{code}

  \funparagraph{Color definition}  % The name of the function(s)
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

short _def_bw(technology t, colorcode c, short bwf)

{
  if(nr_of_ccodes<=c) {
    rraise(INVALID_COLOR_CODE);
    return 0;
  }
  switch(t) {
  case BW_SCREEN:
    bwtech[0][c] = bwf;
    break;
  case BW_PAPER:
    bwtech[1][c] = bwf;
    break;
  case BW_SLIDE:
    bwtech[2][c] = bwf;
    break;
  default:
    rraise(INVALID_TECHNOLOGY);
    return 0;
  }
  return 1;
}

short _get_bw(technology t, colorcode c)

{
  if(nr_of_ccodes<=c) {
    rraise(INVALID_COLOR_CODE);
    return -1;
  }
  switch(t) {
  case BW_SCREEN:
    return bwtech[0][c];
  case BW_PAPER:
    return bwtech[1][c];
  case BW_SLIDE:
    return bwtech[2][c];
  default:
    rraise(INVALID_TECHNOLOGY);
    return -1;
  }
}

short _def_gray(technology t, colorcode c, shade s)

{
  if(nr_of_ccodes<=c) {
    rraise(INVALID_COLOR_CODE);
    return 0;
  }
  switch(t) {
  case GRAYS_SCREEN:
    graystech[0][c] = s;
    break;
  case GRAYS_PAPER:
    graystech[1][c] = s;
    break;
  case GRAYS_SLIDE:
    graystech[2][c] = s;
    break;
  default:
    rraise(INVALID_TECHNOLOGY);
    return 0;
  }
  return 1;
}

shade _get_gray(technology t, colorcode c)

{
  if(nr_of_ccodes<=c) {
    rraise(INVALID_COLOR_CODE);
    return -1;
  }
  switch(t) {
  case GRAYS_SCREEN:
    return graystech[0][c];
  case GRAYS_PAPER:
    return graystech[1][c];
  case GRAYS_SLIDE:
    return graystech[2][c];
  default:
    rraise(INVALID_TECHNOLOGY);
    return -1;
  }
}


short _def_cmyk_color(technology t, colorcode c, shade cy, shade ma, shade ye,
		      shade k)

{
  if(nr_of_ccodes<=c) {
    rraise(INVALID_COLOR_CODE);
    return 0;
  }
  switch(t) {
  case COLOR_SCREEN:
    ctech[0][c] = cy;
    mtech[0][c] = ma;
    ytech[0][c] = ye;
    ktech[0][c] = k;
    break;
  case COLOR_PAPER:
    ctech[1][c] = cy;
    mtech[1][c] = ma;
    ytech[1][c] = ye;
    ktech[1][c] = k;
    break;
  case COLOR_SLIDE:
    ctech[2][c] = cy;
    mtech[2][c] = ma;
    ytech[2][c] = ye;
    ktech[2][c] = k;
    break;
  default:
    rraise(INVALID_TECHNOLOGY);
    return 0;
  }
  return 1;
}

shade _get_cyan(technology t, colorcode c)

{
  if(nr_of_ccodes<=c) {
    rraise(INVALID_COLOR_CODE);
    return -1;
  }
  switch(t) {
  case COLOR_SCREEN:
    return ctech[0][c];
  case COLOR_PAPER:
    return ctech[1][c];
  case COLOR_SLIDE:
    return ctech[2][c];
  default:
    rraise(INVALID_TECHNOLOGY);
    return -1;
  }
}


shade _get_magenta(technology t, colorcode c)

{
  if(nr_of_ccodes<=c) {
    rraise(INVALID_COLOR_CODE);
    return -1;
  }
  switch(t) {
  case COLOR_SCREEN:
    return mtech[0][c];
  case COLOR_PAPER:
    return mtech[1][c];
  case COLOR_SLIDE:
    return mtech[2][c];
  default:
    rraise(INVALID_TECHNOLOGY);
    return -1;
  }
}


shade _get_yellow(technology t, colorcode c)

{
  if(nr_of_ccodes<=c) {
    rraise(INVALID_COLOR_CODE);
    return -1;
  }
  switch(t) {
  case COLOR_SCREEN:
    return ytech[0][c];
  case COLOR_PAPER:
    return ytech[1][c];
  case COLOR_SLIDE:
    return ytech[2][c];
  default:
    rraise(INVALID_TECHNOLOGY);
    return -1;
  }
}


shade _get_k(technology t, colorcode c)

{
  if(nr_of_ccodes<=c) {
    rraise(INVALID_COLOR_CODE);
    return -1;
  }
  switch(t) {
  case COLOR_SCREEN:
    return ktech[0][c];
  case COLOR_PAPER:
    return ktech[1][c];
  case COLOR_SLIDE:
    return ktech[2][c];
  default:
    rraise(INVALID_TECHNOLOGY);
    return -1;
  }
}


/*--
  \end{code}

  \section{Hardcopy implementation}

  \ImplementationNotes{
  % How implementation of the whole package is done: algorithms---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

static short eps_unit = 1000;    /* i.e. one pixel per PostScript point */


/*--
  \end{code}

  \funparagraph{wi\_set\_eps\_unit, wi\_get\_eps\_unit}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

void wi_set_eps_unit(short u)

{
  if(u>0)
    eps_unit = u;
}

short wi_get_eps_unit()

{
  return eps_unit;
}


/*--
  \end{code}

  \funparagraph{wi\_print\_dwin}  % The name of the function(s)
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  \verb|_print_dwin| is the main function, called under various
  contexts by all the hardcopy primitives. This function assumes that
  the bounding box has been correctly set, so that it can assimilate
  the PostScript coordinate origin with the WIPORT lower left corner,
  that the scaling of the points was done so that it can consider
  one PostScript point to be equal to one pixel, that the current font
  was set, it is a fixed space font and the size of the
  font corresponds to \verb|font_x_size| and \verb|font_y_size|.
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

short _print_color_defs(FILE *psf)

{
  short i;

  for(i=0; i<nr_of_ccodes; i++) {
    switch(crt_hardcopy_tech) {
    case BW_PAPER:
      fprintf(psf, "/pen-%d { %d setgray } def\n",
	      i, (bwtech[1][i]==0) ? 1 : 0);
      break;
    case BW_SLIDE:
      fprintf(psf, "/pen-%d { %d setgray } def\n",
	      i, (bwtech[2][i]==0) ? 1 : 0);
      break;
    case GRAYS_PAPER:
      fprintf(psf, "/pen-%d { 1.0 %d 1000 div sub setgray } def\n",
	      i, graystech[1][i]);
      break;
    case GRAYS_SLIDE:
      fprintf(psf, "/pen-%d { 1.0 %d 1000 div sub setgray } def\n",
	      i, graystech[2][i]);
      break;
    case COLOR_PAPER:
      fprintf(psf, "/pen-%d { %d 1000 div %d 1000 div %d 1000 div %d 1000 div",
	      i, ctech[1][i], mtech[1][i], ytech[1][i], ktech[1][i]);
      fprintf(psf, " setcmykcolor} def\n");
      break;
    case COLOR_SLIDE:
      fprintf(psf, "/pen-%d { %d 1000 div %d 1000 div %d 1000 div %d 1000 div",
	      i, ctech[2][i], mtech[2][i], ytech[2][i], ktech[2][i]);
      fprintf(psf, " setcmykcolor} def\n");
      break;
    }
  }
}


short _print_dwin(FILE *psf, gwin g, dwin d)

{
  GWIN *gg;
  short i, j, xx, yy;
  short xxl, yyl;
  short crtcolor;

  crtcolor = -1;
  fprintf(psf, "1 setlinewidth\n");
  gg = gwins[g];

  /* prepare the graphic context */
  
  fprintf(psf, "gsave\n");
  xx = gg->dw[d]->x0;
  yy = gg->dw[0]->yl - gg->dw[d]->y0;
  
   /*
  fprintf(psf, "newpath %d %d moveto %d %d rlineto %d %d rlineto\n",
    xx, yy, gg->dw[d]->xl, 0, 0, - gg->dw[d]->yl);
  fprintf(psf, "  %d %d rlineto %d %d rlineto closepath clip\n",
    - gg->dw[d]->xl, 0, 0, gg->dw[d]->yl);
   */
   
  for(i=0; i<gg->dw[d]->gpfill; i++)
    for(j=0;
	j<((i==gg->dw[d]->gpfill-1) ? gg->dw[d]->gppfill : NOGPERPAGE);
	j++) {

      /* compute object placement position */

      xx = gg->dw[d]->x0 + gg->dw[d]->gp[i][j].x0;
      yy = gg->dw[d]->y0 + gg->dw[d]->gp[i][j].y0;
      yy = gg->dw[0]->yl - yy;  /* Different coordinate placement in PS */

      /* change color if necessary */

      if(gg->dw[d]->gp[i][j].gcolor!=crtcolor) 
	fprintf(psf, "pen-%d ", crtcolor = gg->dw[d]->gp[i][j].gcolor);
      switch(gg->dw[d]->gp[i][j].gtype) {
      case TEXTGREL:
	fprintf(psf, "\n%%  TextGrel %d:%d\n", i, j);
	fprintf(psf, "%d %d moveto (%s) show\n",
		xx, yy-15, gg->dw[d]->gp[i][j].ext);
	break;
      case LINEGREL:
	fprintf(psf, "newpath %d %d moveto %d %d lineto stroke\n",
		xx, yy,
		xx+(gg->dw[d]->gp[i][j].x1 - gg->dw[d]->gp[i][j].x0),
		yy+(gg->dw[d]->gp[i][j].y0 - gg->dw[d]->gp[i][j].y1));
	break;
      case TEXTPADGREL:
	fprintf(psf, "\n%%  TextPadGrel %d:%d\n", i, j);
        fprintf(psf, "\npen-%d\n", wi_widget_darkborder);
      	xxl = gg->dw[d]->gp[i][j].x1 - gg->dw[d]->gp[i][j].x0;
	yyl = gg->dw[d]->gp[i][j].y0 - gg->dw[d]->gp[i][j].y1;
	fprintf(psf, "newpath %d %d moveto %d %d lineto %d %d lineto\n",
		xx, yy, xx+xxl, yy, xx+xxl, yy+yyl);
	fprintf(psf, "        %d %d lineto %d %d lineto closepath stroke\n",
		xx, yy+yyl, xx, yy);
	fprintf(psf, "%d %d moveto (%s) show\n",
		xx, yy-15, gg->dw[d]->gp[i][j].ext);
	break;
      case SWITCHGREL:
      case ASWITCHGREL:
	fprintf(psf, "\n%%  Switch or AltSwitch %d:%d\n", i, j);
        fprintf(psf, "\npen-%d\n", wi_widget_darkborder);
      	xxl = gg->dw[d]->gp[i][j].x1 - gg->dw[d]->gp[i][j].x0;
	yyl = gg->dw[d]->gp[i][j].y0 - gg->dw[d]->gp[i][j].y1;

	yy -= 15;

	/* print the empty switch */
	fprintf(psf, "newpath %d %d moveto %d %d lineto %d %d lineto\n",
		xx, yy, xx+10, yy, xx+10, yy+10);
	fprintf(psf, "        %d %d lineto %d %d lineto closepath stroke\n",
		xx, yy+10, xx, yy);

        /* print the filling if appropiate */
	if(gg->dw[d]->gp[i][j].gkey == gg->dw[d]->gp[i][j].swvval) {
	  fprintf(psf, "newpath %d %d moveto %d %d lineto %d %d lineto\n",
		  xx+2, yy+2, xx+8, yy+2, xx+8, yy+8);
	  fprintf(psf, "        %d %d lineto %d %d lineto closepath fill\n",
		  xx+2, yy+8, xx+2, yy+2);
	}

	fprintf(psf, "%d %d moveto (%s) show\n",
		xx + 14, yy, gg->dw[d]->gp[i][j].ext);
	break;
      case BUTTONGREL:
	fprintf(psf, "\n%%  ButtonGrel %d:%d\n", i, j);
	xxl = gg->dw[d]->gp[i][j].x1 - gg->dw[d]->gp[i][j].x0;
	yyl = gg->dw[d]->gp[i][j].y0 - gg->dw[d]->gp[i][j].y1;
	fprintf(psf, "%d %d moveto (%s) show\n",
		xx + (xxl - wi_get_char_x_size() * strlen(gg->dw[d]->gp[i][j].ext))/2,
		yy - 13, gg->dw[d]->gp[i][j].ext);

        fprintf(psf, "pen-%d\n", wi_widget_darkborder);
	fprintf(psf, "newpath %d %d moveto %d %d lineto %d %d lineto\n",
		xx, yy, xx+xxl, yy, xx+xxl, yy+yyl);
	fprintf(psf, "        %d %d lineto %d %d lineto closepath stroke\n",
		xx, yy+yyl, xx, yy);

	fprintf(psf, "newpath %d %d moveto %d %d lineto %d %d lineto stroke\n",
		xx+1, yy+yyl-1, xx+xxl+1, yy+yyl-1, xx+xxl+1, yy-1);
	fprintf(psf, "newpath %d %d moveto %d %d lineto %d %d lineto stroke\n",
		xx+2, yy+yyl-2, xx+xxl+2, yy+yyl-2, xx+xxl+2, yy-2);

	break;
      default:
	break;
      }
    }
   fprintf(psf, "grestore\n\n");
}


short wi_print_dwin(char *fname, dwin d)

{
  FILE *psf;
  GWIN *gg;

  psf = fopen(fname, "w");
  if(psf==NULL) /* file open error */
    return 0;

  gg = gwins[wi_get_crt_gwin()];
  fprintf(psf, "%%!PS-Adobe-2.0 EPSF-2.0\n");
  fprintf(psf, "%%%%Title: %s\n", fname);
  fprintf(psf, "%%%%Creator: %d\n", wi_app_name);
  fprintf(psf, "%%%%BoundingBox: %d %d %d %d\n\n",
	  gg->dw[d]->x0 * wi_get_eps_unit() / 1000.0,
          0,
	  (gg->dw[d]->x0 + gg->dw[d]->xl) * wi_get_eps_unit() / 1000.0,
	  (gg->dw[0]->yl - gg->dw[d]->y0) * wi_get_eps_unit() / 1000.0);
  fprintf(psf, "%%%%Pages: 1\n");
  fprintf(psf, "%%%%DocumentFonts:\n");
  fprintf(psf, "%%%%EndComments\n");
  fprintf(psf, "%%%%EndProlog\n\n");
  fprintf(psf, "%%%%Page: 1 1\n\n");
  fprintf(psf, "%%  Contents of the whole dwin %d of gwin %d\n\n", 
    d, wi_get_crt_gwin());
  fprintf(psf, " %d dup 1000.0 div exch 1000.0 div scale\n",
	  wi_get_eps_unit());
  _print_color_defs(psf);
  fprintf(psf, "\n/Helvetica findfont 14 scalefont setfont\n\n");
  _print_dwin(psf, wi_get_crt_gwin(), d);
  fprintf(psf, "\n%%End of EPS file\n\n");
  fclose(psf);
  
}


/*--
  \end{code}

  \funparagraph{wi\_print\_gwin}  % The name of the function(s)
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/


short wi_print_gwin(char *fname)

{
  FILE *psf;
  GWIN *gg;
  short d;

  psf = fopen(fname, "w");
  if(psf==NULL) /* file open error */
    return 0;

  gg = gwins[wi_get_crt_gwin()];
  fprintf(psf, "%%!PS-Adobe-2.0 EPSF-2.0\n");
  fprintf(psf, "%%%%Title: %s\n", fname);
  fprintf(psf, "%%%%Creator: %d\n", wi_app_name);
  fprintf(psf, "%%%%BoundingBox: %d %d %d %d\n\n",
	  (int)(gg->dw[0]->x0 * wi_get_eps_unit() / 1000.0),
	  0,
	  (int)((gg->dw[0]->x0 + gg->dw[0]->xl) * wi_get_eps_unit() / 1000.0),
	  (int)((gg->dw[0]->yl - gg->dw[0]->y0) * wi_get_eps_unit() / 1000.0));
  fprintf(psf, "%%%%Pages: 1\n");
  fprintf(psf, "%%%%DocumentFonts:\n");
  fprintf(psf, "%%%%EndComments\n");
  fprintf(psf, "%%%%EndProlog\n\n");
  fprintf(psf, "%%%%Page: 1 1\n\n");
  fprintf(psf, "%%  Contents of the whole gwin %d\n\n", wi_get_crt_gwin());
  
  fprintf(psf, " %d dup 1000.0 div exch 1000.0 div scale\n",
	  wi_get_eps_unit());
  _print_color_defs(psf);
  fprintf(psf, "\n/Helvetica findfont 14 scalefont setfont\n\n");
  for(d=0; d<gg->dwinfill; d++)
    _print_dwin(psf, wi_get_crt_gwin(), d);
  fprintf(psf, "\n%%End of EPS file\n\n");
  fclose(psf);
  
}


/*--
\end{code}

\section{textpad support} 

\ImplementationNotes{ 

}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

\begin{code}
--*/

int _tp_init( struct _tp_ext * this, char * initialcontents
			  , int size )  {

  this->contents = malloc(size + 1);
  if (this->contents == NULL) {
    gwerrno = NO_MEMORY_FOR_TEXT;
    return 0;
  }
  strncpy(this->contents, initialcontents, size); 
  this->contents_fill = strlen(this->contents);
  this->contents_size = size;
  this->crs = -1;		/* no cursor in the textpad */
  return 1;
}

void _tp_free( struct _tp_ext * this ) {
  free( this->contents );
  this->contents_size = this->contents_fill = 0;
  this->crs=-1;
}

/*--
\end{code}

\subsection{textpad} % The package name, and the fact it is the body here

\ImplementationNotes{ }
\Portability{ }
\Changes{ }

\begin{code}
--*/


/*--
\end{code}

\funparagraph{textpad\_format\_buffer}
\PurposeReturn{The number of lines found in the buffer but not
exceeding sol\_max. The sol array will hold indices
to  the start of each line. The graphical representation of any line
will be no larger than the given width, lines will eventually be 
wrapped around at spaces.}
\Exceptions{}
\GlobalEffects{}
\Preconditions{}
\Performance{}

\begin{code}
--*/

static int textpad_format_buffer( char * buffer
			  , short * sol, int sol_max 
			  , int width ){
  int lines;			/* the actual number of lines */
  int position;			/* current position in the buffer */
  int length;			/* length of the string */
  int line_width;		/* current line width  */
  int break_pos;		/* position of a possible break */

  /* initialization */
  sol[0] = 0;
  length = strlen( buffer );
  if( length == 0 ) return 0;

  line_width = position = 0;
  break_pos = -1;
  lines = 1;

  while((lines<sol_max) && (position < length) ) {
    /* remember last possible break position */
    if ( buffer[position] == ' ' ) break_pos=position;
    /* got to chose a line break */
    if( line_width >= width ) {	
      if( break_pos!=-1 ) {
	/* wind back to the previous break position */
	position = break_pos + 1;
      }
      sol[lines] = position;
      
      /* resetting state for the beginning of a new line */
      break_pos = -1;
      lines++;
      line_width = 0;
    } else {
      /* normal line processing */
      position ++;
      line_width += wi_get_char_x_size();    
    }
  }
  /* save a pointer to the end of the last line */
  if(position==length) 
    sol[lines] = position;
  return lines;
}


/*--
\end{code}

\funparagraph{textpad\_cursor\_line}
\PurposeReturn{ The textpad line on which the cursor will be placed.}
\Exceptions{}
\GlobalEffects{}
\Preconditions{}
\Performance{}

\begin{code}
--*/


static int textpad_cursor_line( int crs, short * sol, int line_nb) {
  int line;
  
  if( crs == 0 ) return 0;
  for( line=0; line<line_nb; line++ ) 
    if( sol[line]>crs ) {
      break;
    } 

  return line-1;
}


/*--
\end{code}

\funparagraph{\_tp\_process\_key}  % The name of the function(s)
\ImplementationNotes{}
\Portability{}
\Changes{}

\begin{code}
--*/

void _tp_format( struct _tp_ext * this, short width ) {
  this->line_nb = textpad_format_buffer( this->contents
					 , this->sol, 50, width );
  this->cursor_line = textpad_cursor_line( this->crs
					  , this-> sol, this->line_nb);
}


void _tp_select( struct _tp_ext * this, short width ) {
  if( this->crs == -1 ) {
    /* first call initialize */
    this->crs = 0;
    _tp_format( this, width );
  }
}

void _tp_end_selection( struct _tp_ext * this ) {
 this->crs=-1;
}

int _tp_process_key( struct _tp_ext * this, long thekey, short width ) {
  int i;


  printf("Processing key %4x [%c]\n", thekey, thekey); 
  switch(thekey) {
  case 13:       /* in fact this should never happen :-)*/
  case 10:
    this->crs=-1;
    return 1;
    break;
  case DEL:
    if( this->contents_fill==0)  break;
    for(i=(this->crs); i<=this->contents_fill; i++)
      this->contents[i] = this->contents[i+1];
    this->contents_fill --;
    _tp_format( this, width );
    break;
  case C_K:
    this->contents_fill = 0;
    this->contents[0] = 0;
    this->crs = 0;
    _tp_format(this, width);
    break;
  case BS:
    if( this->contents_fill==0)  break;
    if( this->crs == 0 ) break;
    (this->crs) --;
    for(i=(this->crs); i<=this->contents_fill; i++)
      this->contents[i] = this->contents[i+1];
    this->contents_fill --;
    _tp_format( this, width );
    break;
  case LEFT:
    if (this->crs) { 
      (this->crs) --;
      this->cursor_line 
        = textpad_cursor_line( this->crs, this->sol, this->line_nb );
    }
    break;
  case RIGHT:
    if( this->crs < this->contents_fill) {
      (this->crs)++;
      this->cursor_line 
        = textpad_cursor_line( this->crs, this->sol, this->line_nb );
    }
    break;
  case HOME:
    this->crs = 0;
    this->cursor_line 
      = textpad_cursor_line( this->crs, this->sol, this->line_nb );
    break;
  case END:
    this->crs = this->contents_fill;
    this->cursor_line 
      = textpad_cursor_line( this->crs, this->sol, this->line_nb );
    break;
  case UP:
    if( this->cursor_line > 0 ) {
      this->crs = this->crs - this->sol[this->cursor_line];
      this->cursor_line --;
      this->crs += this->sol[this->cursor_line];
    }
    break;
   case 0: 
    break;
  case DOWN:
    if( this->cursor_line < this->line_nb-1 ) {
      this->crs = this->crs - this->sol[this->cursor_line];
      this->cursor_line ++;
      this->crs += this->sol[this->cursor_line];
    }
    break;
  default:
    if(thekey<256) {
      if(this->contents_fill+1 < this->contents_size) {
	for(i=this->contents_fill+2; i>this->crs; i--)
	  this->contents[i] = this->contents[i-1];
	this->contents[this->crs] = thekey;
      }
      (this->crs) ++;
      this->contents_fill++;
      _tp_format( this, width );
      break;
    }
  }
  return 0;
}


/*--
\end{code}

\funparagraph{\_tp\_contents}
\PurposeReturn{}
\Exceptions{}
\GlobalEffects{}
\Preconditions{}
\Performance{}

\begin{code}
--*/


void _tp_contents( struct _tp_ext * this, char * newcontents, short width ) {
  /* function to be written */
  strncpy( this->contents, newcontents, this->contents_size );
  _tp_format( this, width);
}



/*--
\end{code}

\funparagraph{_tp_access_contents}  % The name of the function(s)
\ImplementationNotes{ 
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

\begin{code}
--*/

char * _tp_access_contents( struct _tp_ext * this ) {
 return this->contents;
}

/*--
\end{code}

\funparagraph{\_tp\_cursor}  % The name of the function(s)
\ImplementationNotes{}
\Portability{}
\Changes{}

\begin{code}
--*/

int _tp_cursor_line( struct _tp_ext * this ){
  if( this->crs > -1 ) 
    return this->cursor_line;
  else
    return -1;
}

int _tp_cursor_row( struct _tp_ext * this ){
  if( this-> crs > -1 ) 
    return this->crs-this->sol[this->cursor_line];
  else 
    return -1;
}

char * _tp_access_line( struct _tp_ext * this, short line) { 
  return this->contents+this->sol[line]; 
}

short _tp_line_length( struct _tp_ext * this, short line ) {
  return this->sol[line+1] - this->sol[line];
}


/*--
\end{code}

\section{get_selection} % The package name, and the fact it is the body here

\ImplementationNotes{ 
A linear search is performed over all the registered grels.
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

\begin{code}
--*/
int _is_selectable_grel( struct _grel * g ) {
  switch( g->gtype ) {
   case BUTTONGREL:
   case TEXTPADGREL:
   case BUFFERGREL:
    return 1;
  default:
    return 0;
  }
}

#define is_input_grel(g) (g->gtype==BUTTONGREL || g->gtype=SWITCHGREL || \
                          g->gtype==ASWITCHGREL || g->gtype==TEXTPADGREL)

void get_position_selection(struct selection *  sel, GWIN * gw
				      , short x, short y) { /* [FUN] */ 
  /* fill up the sel argument with the grel and dwin located at
   * the given position
   */
  int d;
  ITERATOR it;
  struct _grel * g;
  short local_x, local_y; /* position in the current dwin */
  
#define GREL_POS_ACTIVATED(g, x, y) ( g-> x0 <= x && g->x1 >= x && \
                                     g->y0 <= y && g->y1 >= y)
  /* initialization */
  sel->lasthitdwin = NULL;
  sel->hitgrel = NULL;
  /* for all the dwins of the gwin */
  for (d = gw -> dwinfill - 1; d >= 0; d --) {
    local_x = x - gw->dw[d]->x0;
    local_y = y - gw->dw[d]->y0;
    init_bwd_grel_iterator(gw, d, &it);
    while (check_lower_bound_grel(&it)) {
      g = access_grel(&it);
      if ((g->gtype==BUTTONGREL || g->gtype==SWITCHGREL || 
	   g->gtype==ASWITCHGREL || g->gtype==TEXTPADGREL)
	  && GREL_POS_ACTIVATED(g, local_x, local_y)) {
	sel->lasthitdwin = gw->dw[d];
	sel->hitgrel=g;
	sel->lasthitdwin_ndx = d;
      }
      iterator_backward(&it);
    }
  }
}

void get_key_selection(struct selection * sel
			   , GWIN * gw, short key) { /* [FUN] */ 
  /* fill up the sel argument with the grel and dwin selected
   * by the given key
   */
  int d;
  ITERATOR it;
  struct _grel * g;
  
#define GREL_KEY_ACTIVATED(g, key) (g -> gkey == key) /* [DEF] */ 
  /* initialization */
  sel->lasthitdwin = NULL;
  sel->hitgrel = NULL;
  /* for all the dwins of the gwin */
  for (d = gw -> dwinfill - 1; d >= 0; d --) {
    init_bwd_grel_iterator(gw, d, &it);
    while (check_lower_bound_grel(&it)) {
      g = access_grel(&it);
      if (GREL_KEY_ACTIVATED(g, key)) {
	sel->lasthitdwin = gw->dw[d];
	sel->hitgrel=g;
	sel->lasthitdwin_ndx = d;
	return;
      }
      iterator_backward(&it);
    }
  }
}



/*--
\end{code}

\section{On line help} 

\ImplementationNotes{ 
The correspondence between a grel and the associated help string is
held in a hash table.
}
\Portability{}
\Changes{}

\begin{code}
--*/


/*--
\end{code}

\subsection{Hash table}

\ImplementationNotes{ Adapted from the VTSa 1.3 implementation}
\Portability{}
\Changes{}

\begin{code}
--*/
/*
 * hash.c
 *	A hashed lookup mechanism
 */

/*
 * This is the root of the hashed object.  You shouldn't monkey
 * with it directly.
 */
struct hash {
	int h_hashsize;		/* Width of h_hash array */
	struct hash_node	/* Chains under each hash value */
		*h_hash[1];
};

/*
 * Hash collision chains.  An internal data structure.
 */
struct hash_node {
	struct hash_node *h_next;	/* Next on hash chain */
	long h_key;			/* Key for this node */
	void *h_data;			/*  ...corresponding value */
};


/*
 * Hash routines
 */
static struct hash *hash_alloc(int);
static int hash_insert(struct hash *, long, void *);
static int hash_delete(struct hash *, long);
static void *hash_lookup(struct hash *, long);
static void hash_dealloc(struct hash *);
static unsigned hash_size(struct hash *);

/* implementation */
/*
 * hashval()
 *	Convert key into hash value
 */
static unsigned
hashidx(unsigned long key, unsigned size)
{
	return((key ^ (key >> 4)) % size);
}

/*
 * hash_alloc()
 *	Allocate a hash data structure of the given hash size
 */
static struct hash *
hash_alloc(int hashsize)
{
	struct hash *h;

	h = malloc(sizeof(struct hash) + hashsize*sizeof(struct hash *));
	if (h) {
		h->h_hashsize = hashsize;
		bzero(&h->h_hash, hashsize*sizeof(struct hash *));
	}
	return(h);
}

/*
 * hash_insert()
 *	Insert a new key/value pair into the hash
 *
 * Returns 1 on error, 0 on success.
 */
static hash_insert(struct hash *h, long key, void *val)
{
	struct hash_node *hn;
	unsigned idx;

	if (!h) {
		return(1);
	}
	idx = hashidx(key, h->h_hashsize);
	hn = malloc(sizeof(struct hash_node));
	if (!hn) {
		return(1);
	}
	hn->h_key = key;
	hn->h_data = val;
	hn->h_next = h->h_hash[idx];
	h->h_hash[idx] = hn;
	return(0);
}

/*
 * hash_delete()
 *	Remove node from hash
 *
 * Returns 1 if key not found, 0 if removed successfully.
 */
static hash_delete(struct hash *h, long key)
{
	struct hash_node **hnp, *hn;
	unsigned idx;

	if (!h) {
		return(1);
	}

	/*
	 * Walk hash chain list.  Walk both the pointer, as
	 * well as a pointer to the previous pointer.  When
	 * we find the node, patch out the current node and
	 * free it.
	 */
	idx = hashidx(key, h->h_hashsize);
	hnp = &h->h_hash[idx];
	hn = *hnp;
	while (hn) {
		if (hn->h_key == key) {
			*hnp = hn->h_next;
			free(hn);
			return(0);
		}
		hnp = &hn->h_next;
		hn = *hnp;
	}
	return(1);
}

/*
 * hash_dealloc()
 *	Free up the entire hash structure
 */
static void
hash_dealloc(struct hash *h)
{
	unsigned x;
	struct hash_node *hn, *hnn;

	for (x = 0; x < h->h_hashsize; ++x) {
		for (hn = h->h_hash[x]; hn; hn = hnn) {
			hnn = hn->h_next;
			free(hn);
		}
	}
	free(h);
}

/*
 * hash_lookup()
 *	Look up a node based on its key
 */
static void *
hash_lookup(struct hash *h, long key)
{
	struct hash_node *hn;
	unsigned idx;

	if (!h) {
		return(0);
	}
	idx = hashidx(key, h->h_hashsize);
	for (hn = h->h_hash[idx]; hn; hn = hn->h_next) {
		if (hn->h_key == key) {
			return(hn->h_data);
		}
	}
	return(0);
}

/*
 * hash_size()
 *	Tell how many elements are stored in the hash
 */
static unsigned
hash_size(struct hash *h)
{
	unsigned x, cnt = 0;
	struct hash_node *hn;

	for (x = 0; x < h->h_hashsize; ++x) {
		for (hn = h->h_hash[x]; hn; hn = hn->h_next) {
			cnt += 1;
		}
	}
	return(cnt);
}

/*--
\end{code}

\section{On line help --- interface functions}

\ImplementationNotes{}
\Portability{}
\Changes{}

\begin{code}
--*/

#define GW_HELP(gw)     ((struct hash*)(gw)->help_private)

void wi_set_help( ugrel g, char * h ) {
  hash_insert( GW_HELP(wi_crt_gwin_ptr()), g, h );
}

long wi_set_area_help( dwin d, pixed x0, pixed y0, pixed xl, pixed yl ){
  /* FUNCTION NOT YET IMPLEMENTED */
  return 0;
}

void wi_del_help(  ugrel g ) {
  hash_delete( GW_HELP(wi_crt_gwin_ptr()), g );
}

void wi_def_help_dwin( ugrel d ) {
  GWIN * gw;

  gw =  wi_crt_gwin_ptr();
  if( gw->help_private == NULL ) {
    /* initialize help specific information */
    gw->help_private = hash_alloc( 128 );
  }
  gw->help_dwin = d;
  gw->help_on = 1;
}

void wi_set_help_on( ) {
  wi_crt_gwin_ptr() -> help_on = 1;
}

void wi_set_help_off( ) {
  wi_crt_gwin_ptr() -> help_on = 0;
}


/*--
\end{code}

\subsection{On line help --- internal functions}

\ImplementationNotes{  }
\Portability{}
\Changes{}

\begin{code}
--*/

char * _locate_help(  GWIN * gw ,  short x, short y ) {
  ugrel u;
  struct _grel * gr;
  struct _dwin * dw;
  unsigned ndx;
  struct hash * h;
  struct hash_node *hn;

  h = GW_HELP(gw);
  for (ndx = 0; ndx < h->h_hashsize; ++ndx) {
    for (hn = h->h_hash[ndx]; hn; hn = hn->h_next) {
      u = hn->h_key;
      dw = gw->dw[dwin_of_ugrel(u)];
      gr = ndx_to_grel(dw, grel_of_ugrel(u));
      if (GREL_POS_ACTIVATED(gr, x - dw->x0, y - dw->y0)) 
	return hn->h_data;
    }
  }
  return NULL;			/* no help string at this position */
}


/*--
  \end{code}

  % MOVE THIS COMMENT TO THE END OF THE FILE
  \endoffile

  --*/
