/*    WIPort 2 for X11
 *
 *    Copyright (c) 1993--1998 Alexandru Dan Corlan, Marius Seritan
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU Library General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    In short, you can use and modify this program. You can
 *    distribute the program with or without your modifications if you want,
 *    but you must not require licence fees and must provide the
 *    full source code with the distribution. If you make an application program
 *    which just calls functions from this library you may distribute that
 *    application program without source code, but also as an linkable object file, 
 *    and must distribute also this library with source code.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Library General Public License for more details.
 *
 *    You should have received a copy of the GNU Library General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This module is the implementation of the WIPORT user
 * interface solution, Level 1, for the X Window system,
 * using only the Xlib functions and resources.
 * It is written in plain C (ANSI-C) for
 * maximum portability and was first developed on the
 * Linux V0.99pl2 operating system using the 
 * GNU-CC compiler version 2.2.2 for the Intel 80386 
 * microprocessor.
 * 
 * This file falls under the regulations of the GNU licence
 * for libraries, version 1.
 *
 * 8 Mars 1995 mseritan: starting to upgrade the library to Wiport 2nd version
 *        1. removed functions duplicated by wipall.o
 */

#include <stdio.h>
#include <stdlib.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>

#include "wiport2.h"

#define PAINTNOW

short gwerrno;

long Black;
long White;

short maxgwinxsize;
short maxgwinysize;
short maxgwindepth;

short font_x_size = 8; /* there will be a single font */
short font_y_size = 16; 
short wi_get_char_x_size() { return font_x_size;}
short wi_get_char_y_size() { return font_y_size;}

/* colors of the various widgets */

short wi_widget_background = 8;
short wi_widget_darkborder = 4;
short wi_widget_lightborder = 12;

Cursor thecursor;
XFontStruct * thefont;

/* variables for shots */
short xspotshot;
short yspotshot;
short xhorshot;
short y0horshot;
short y1horshot;
short x0vershot;
short x1vershot;
short yvershot;
short x0rectshot;
short x1rectshort;
short y0rectshot;
short y1rectshot;

inline void cursor_hide(void) {
    GrMouseEraseCursor();
}
inline void cursor_show(void) {
    GrMouseDisplayCursor();
}

/*  communication variables */
Display *crtdisplay = NULL;
int crtscreen;

/* data structures used for framed boxes ---> Motif like buttons */

typedef struct {
  long fbx_intcolor;
  long fbx_topcolor;
  long fbx_leftcolor;
  long fbx_rightcolor;
  long fbx_bottomcolor;
}  GrFBoxColors;


static GrFBoxColors butup; /* [VAR] */ 
static GrFBoxColors butdown; /* [VAR] */ 
static GrFBoxColors switchno; /* [VAR] */ 
static GrFBoxColors switchyes; /* [VAR] */ 
static GrFBoxColors textpd;    /* [VAR] */
static GrFBoxColors lowdwin; /* [VAR] */ 
static GrFBoxColors raisedwin;  /* [VAR] */ 
static GrFBoxColors planedwin;  /* [VAR] */ 

static short last_line_grel = -1;

/*--
\end{code}

\funparagraph{FramedBox}  % The name of the function(s)
\ImplementationNotes{ }
\Portability{ }
\Changes{ }

\begin{code}
--*/


/* libgrx like functions */
static GC framedbox_gc = NULL;
static GC left_gc;
static GC right_gc;
static GC top_gc;
static GC bottom_gc;

void FramedBox(GWIN * gw, int x1,int y1
		 , int x2,int y2,int wdt,GrFBoxColors *c) {
  
  if( framedbox_gc == NULL ) {
    framedbox_gc = XCreateGC (gw->gwid, gw->gwiw, 0, 0);
    left_gc = XCreateGC (gw->gwid, gw->gwiw, 0, 0);
    right_gc = XCreateGC (gw->gwid, gw->gwiw, 0, 0);
    top_gc = XCreateGC (gw->gwid, gw->gwiw, 0, 0);
    bottom_gc = XCreateGC (gw->gwid, gw->gwiw, 0, 0);
  }

  XSetBackground(gw->gwid, framedbox_gc, gw->gwibackground);
  XSetForeground(gw->gwid, framedbox_gc,  c->fbx_intcolor );
  XSetForeground( gw->gwid, top_gc, c->fbx_topcolor );
  XSetForeground( gw->gwid, left_gc, c->fbx_leftcolor );
  XSetForeground( gw->gwid, right_gc, c->fbx_rightcolor );
  XSetForeground( gw->gwid, bottom_gc, c->fbx_bottomcolor  );

  XFillRectangle( gw->gwid, gw->gwiw, framedbox_gc, x1, y1, x2-x1, y2-y1 );


  while(--wdt >= 0) {
    x1--; x2++;
    y1--; y2++;
    XDrawLine( gw->gwid, gw->gwiw,  top_gc, x1 , y1, x2, y1 );
    XDrawLine( gw->gwid, gw->gwiw, left_gc, x1, (y1 + 1),x1, (y2 - 1));
    XDrawLine( gw->gwid, gw->gwiw, right_gc, x2, (y1 + 1), x2, (y2 - 1));
    XDrawLine( gw->gwid, gw->gwiw, bottom_gc, x1, y2, x2, y2);
  }
}


short theargc;  /* to fetch the ``standard properties'' from it */
char **theargv; /* same reason */

long getcolor(Display * dpy, int screen, char * name ) {
  XColor color1, color2;
  XLookupColor( dpy, DefaultColormap(dpy, screen)
	       , name, &color1, &color2 );
  XAllocColor( dpy, DefaultColormap(dpy, screen), &color2 );
  return( color2.pixel );
}


/*--
\end{code}

\funparagraph{wi\_initiate}  % The name of the function(s)
\ImplementationNotes{ 
% How implementation of the function is done: algorithm---if special,
% implementation decisions, discarded implementation alternatives
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

\begin{code}
--*/

short wiportinit(void *display, int screen, short visual_choice,
		 int argc, char **argv) {
  if( display == NULL ) {
    display = XOpenDisplay("");
    if( display == NULL ) {
      printf( "I could not open the display.\n");
      return 0;
    }	
  };
  
  crtdisplay = (Display *)display;
  if(screen==-1)
    crtscreen = DefaultScreen(crtdisplay);
  else
    crtscreen = screen;
  maxgwinxsize = DisplayWidth(crtdisplay,screen);
  maxgwinysize = DisplayHeight(crtdisplay,screen);
  maxgwindepth = DisplayPlanes(crtdisplay,screen); 
  
  /* setting up the colors */
  {
    long lightcolor, mediumcolor, darkcolor;

    Black        = BlackPixel(crtdisplay, crtscreen);
    White        = WhitePixel(crtdisplay, crtscreen);
    
    if( DisplayPlanes( crtdisplay, crtscreen ) == 1 ) { 
      
      /* define colors for the buttons */
      darkcolor = Black;
      mediumcolor = White;
      lightcolor = Black;
    } else {
      printf( "Number of display planes %d.\n"
	     , DisplayPlanes( crtdisplay, crtscreen ));
      
      darkcolor = getcolor(crtdisplay, crtscreen, "Mediumblue");
      mediumcolor = getcolor(crtdisplay, crtscreen, "DodgerBlue");
      lightcolor = getcolor(crtdisplay, crtscreen, "Lightgrey" );
    }

    wi_widget_background = mediumcolor;
    wi_widget_darkborder = darkcolor;
    wi_widget_lightborder = lightcolor;

    butup.fbx_intcolor=mediumcolor;
    butup.fbx_topcolor = butup.fbx_leftcolor =  lightcolor;
    butup.fbx_bottomcolor = butup.fbx_rightcolor =  darkcolor;
    
    butdown.fbx_intcolor = mediumcolor;
    butdown.fbx_topcolor = butdown.fbx_leftcolor =  darkcolor;
    butdown.fbx_bottomcolor = butdown.fbx_rightcolor =  lightcolor;
    
    switchno.fbx_intcolor=mediumcolor;
    switchno.fbx_topcolor = switchno.fbx_leftcolor =  darkcolor;
    switchno.fbx_bottomcolor = switchno.fbx_rightcolor =  lightcolor;
    
    switchyes.fbx_intcolor = Black;
    switchyes.fbx_topcolor = switchyes.fbx_leftcolor =  darkcolor;
    switchyes.fbx_bottomcolor = switchyes.fbx_rightcolor =  lightcolor;
    
    textpd.fbx_intcolor = mediumcolor;
    textpd.fbx_topcolor = textpd.fbx_leftcolor =  darkcolor;
    textpd.fbx_bottomcolor = textpd.fbx_rightcolor =  lightcolor;
    
    lowdwin.fbx_intcolor = mediumcolor;
    lowdwin.fbx_topcolor = lowdwin.fbx_leftcolor =  darkcolor;
    lowdwin.fbx_bottomcolor = lowdwin.fbx_rightcolor =  lightcolor;
    
    raisedwin.fbx_intcolor = mediumcolor;
    raisedwin.fbx_topcolor = raisedwin.fbx_leftcolor =  lightcolor;
    raisedwin.fbx_bottomcolor = raisedwin.fbx_rightcolor =  darkcolor;
  
    planedwin.fbx_intcolor = mediumcolor;
    planedwin.fbx_topcolor = planedwin.fbx_leftcolor =  lightcolor  ;
    planedwin.fbx_bottomcolor = planedwin.fbx_rightcolor =  darkcolor;
  
  }  
  theargc = argc;
  theargv = argv;
  
  /* Loading a suitable font */
  thefont = XLoadQueryFont( crtdisplay
			   , "-*-fixed-*-*-*-*-14-*-*-*-*-*-*-*");
  if( thefont == NULL ) {
    printf( "I couldn't find font.\n" );
    exit( -1 );
  }
  font_x_size = thefont->max_bounds.rbearing - thefont->min_bounds.lbearing;
  font_y_size = thefont->max_bounds.ascent + thefont->max_bounds.descent;
  
  /* defining a nicer cursor */
  {
    thecursor = XCreateFontCursor( crtdisplay, XC_left_ptr);
    if( thecursor == (Cursor)0 ) {
      fprintf( stdout, "Something went wrong with the cursor.\n" );
      exit( 10 );
      
    }
    /*
      XRecolorCursor( crtdisplay, thecursor, White, Black );
      */
  }

  return 1;  /* permanent success */ 
}

short wi_initiate()

{
  wiportinit(NULL, 0, 0, 0, NULL);
  return 1;  /* Success */
}

/*--
  \end{code}

  \funparagraph{wi\_terminate}
  \PurposeReturn{
  Dynamically frees all the dynamically allocated data pertaining to WIPORT.
  }
  \Exceptions{
  }
  \GlobalEffects{
  % Global effects on exported or external variables
  }
  \Preconditions{
  % Conditions to be met when the function is called
  }
  \Performance{
  % How it is, not why and what to do about it
  }

  \begin{code}
  --*/


void wiportclose( void ) {
  /* nothing apropriate */
};

short wi_terminate()

{
  wiportclose(  );
}


void def_screen_color( short index, short red
		      , short green, short blue ) {
  XColor color;
  GWIN * gw;

  gw = wi_crt_gwin_ptr();
  /* set the apropriate information in the structure */
  color.pixel = index;
  color.red = red;
  color.green = green;
  color.blue = blue;
  color.flags =  DoRed | DoGreen | DoBlue;

  XAllocColor( crtdisplay, gw->colormap, &color );
  //  XStoreColor( crtdisplay, gw->colormap, &color);
}


int use_color_s_tech( ) {
  short i;
  GWIN * gw;

#define COLOR_SCALE   65536/16
  for(i=1; i<16; i++)
    def_screen_color(i, i*COLOR_SCALE, i*COLOR_SCALE, i*COLOR_SCALE);
  def_screen_color(0, 8*COLOR_SCALE, 8*COLOR_SCALE, 8*COLOR_SCALE);
#undef COLOR_SCALE
}

/*--
  \end{code}

  \section{Graphic space implementation}

  \ImplementationNotes{
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \funparagraph{wi\_new\_gwin}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{
  The number of colors will currently be the maximum available.
  }

  \begin{code}
  --*/



GWIN *newgwin(short xl, short yl, char *gwititle, char *gwictitle)
     
{
  GWIN *rg;
  short i;
  /* some checks are done here */
  
  if(xl>maxgwinxsize)
    {
      gwerrno = GWIN_X_SIZE_TOO_LARGE;
      return NULL;
    }
  if(yl>maxgwinysize)
    {
      gwerrno = GWIN_Y_SIZE_TOO_LARGE;
      return NULL;
    }
  rg = (GWIN *)malloc(sizeof(struct _gwin));
  if(rg==NULL)
    {
      gwerrno = NO_MEMORY_FOR_GWIN;
      return NULL;
    }
  if(crtdisplay==NULL)
    {
      gwerrno = NO_INITIALIZATION;
      return NULL;
    }
  rg->gwid = crtdisplay;
  rg->gwiscreen = crtscreen;
  /* initialize the help subsystem */
  rg->help_on = -1;
  rg->help_dwin = -1;
  
  /* initialize the X window */
  rg->gwihint.x = ((maxgwinxsize - xl)/2);
  rg->gwihint.y = ((maxgwinysize - yl)/2);
  rg->gwihint.height = yl;
  rg->gwihint.width = xl;
  rg->gwihint.flags = PPosition | PSize;
  rg->gwibackground = White;
  rg->gwiforeground = Black;
  strncpy(rg->gwiconn, gwictitle, GWICONNSIZE);
  strncpy(rg->gwinn, gwititle, GWINNSIZE);
  
  /* Now, let's declare and map this window */
  
  rg->gwiw = XCreateSimpleWindow (rg->gwid, 
				  DefaultRootWindow(rg->gwid),
				  rg->gwihint.x, rg->gwihint.y,
				  rg->gwihint.width, rg->gwihint.height,
				  7, rg->gwiforeground, rg->gwibackground); 
  
  XSetStandardProperties(rg->gwid, rg->gwiw, rg->gwinn,
			 rg->gwiconn, None, theargv, theargc, &(rg->gwihint));
  
  
  rg->gwigc = XCreateGC (rg->gwid, rg->gwiw, 0, 0);
  XSetBackground(rg->gwid, rg->gwigc, rg->gwibackground);
  XSetForeground(rg->gwid, rg->gwigc, rg->gwiforeground);
  {XGCValues temp;
   temp.font = thefont->fid;
   XChangeGC( crtdisplay, rg->gwigc, GCFont, &temp); 
 }
  XDefineCursor( rg->gwid, rg-> gwiw, thecursor );
  XSelectInput(rg->gwid, rg->gwiw
	       , KeyPressMask|ExposureMask|ButtonPressMask
	                     |ButtonReleaseMask|PointerMotionMask);
  
  /* set up the color map */
  rg->colormap = XCreateColormap( rg->gwid, rg->gwiw
				 , DefaultVisual(crtdisplay,crtscreen)
				 , AllocNone);
  XMapRaised (rg->gwid, rg->gwiw);
  rg->clipped_dwin = -1;
  rg->dwinfill = 1;
  for(i=0; i<DWINSPERGWIN; i++)
    rg->dw[i] = NULL;
  rg->dw[0] = dwalloc(0, 0, xl, yl);
  if(rg->dw[0]==NULL)
    {
      free(rg);
      return NULL;
    }
  if(!_add_grel_page(rg->dw[0]))
    {
      free(rg);
      return NULL;
    }
  _register_gwin(rg);
  printf("map created\n");
  return rg;
}


gwin wi_new_gwin(pixed xsize, pixed ysize, short ncolors)

{
  newgwin(xsize, ysize, "Wiport2", "Wiport2");
  return wi_get_crt_gwin();
}


/*--
  \end{code}

  \funparagraph{wi\_set\_gwin\_name}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

short wi_set_gwin_name(char *t)

{
  return 1;
  /* allways success, nothing to be done in a non-windowing environment */
}


/*--
  \end{code}

  \funparagraph{wi\_get\_x\_char\_size, wi\_get\_y\_char\_size}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

pixed wi_get_x_char_size() { return font_x_size; }
pixed wi_get_y_char_size() { return font_y_size; }



/*--
\end{code}

\funparagraph{wi\_color}  % The name of the function(s)
\ImplementationNotes{ 
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

\begin{code}
--*/

void wi_set_color( dwin d, colorcode c ) {
  wi_crt_gwin_ptr()->dw[d]->cr_col = c;
}

colorcode wi_get_color( dwin d ) {
  return( wi_crt_gwin_ptr()->dw[d]->cr_col  );
}

/*--
  \end{code}

  \section{Dwin management}

  \ImplementationNotes{
  % How implementation of the whole package is done: algorithms---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \funparagraph{wi\_new\_dwin}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/



short newdwin(GWIN *gw, short x0, short y0, short xl, short yl)
     
{
  y0 += 5;
  if(gw->dwinfill>=DWINSPERGWIN)
    {
      gwerrno = TOO_MANY_DWINS;
      return -1;
    }
  gw->dw[gw->dwinfill] = dwalloc(x0, y0, xl, yl);
  if(gw->dw[gw->dwinfill]==NULL)
    {
      return -1;
    }
  
  if(!_add_grel_page(gw->dw[gw->dwinfill]))
    {
      free(gw->dw[gw->dwinfill]);
      return -1;
    }
  
  gw->dwinfill ++;
  return (gw->dwinfill - 1);
}


dwin wi_new_dwin(pixed x0, pixed y0, pixed xl, pixed yl)

{
  return newdwin(wi_crt_gwin_ptr(), x0, y0, xl, yl);
}



/*--
  \end{code}

  \funparagraph{wi\_clear\_dwin}  % The name of the function(s)
  \ImplementationNotes{
  We gave up associating dwins with borders of different sizes or
  sculptures. It is probably preferable for the user to draw borders
  himself, in the larger dwin, than to remember the mechanisms involved in
  defining and maintaining dwin borders. Complete coverage of issues
  such as border direction, size, color(s), etc would have complicated
  things a lot. Instead, borders may be associated later with rectangles.

  We gave up having a background color in every dwin, as this will be easy
  to resolve with rectangles filled with the given color. At least two
  more functions are cut this way.
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{
  Background color and optional dwin mapping must be addressed separately.
  }

  \begin{code}
  --*/


void drawborderdwin( GWIN * gw, short d ) {
  int style;
  
  style = gw->dw[d]->style;
  switch(style) {
  case LOWERED_DWIN:
    FramedBox(gw,
	      gw->dw[d]->x0, gw->dw[d]->y0,
	      gw->dw[d]->x0 + gw->dw[d]->xl,
	      gw->dw[d]->y0 + gw->dw[d]->yl, 2, &lowdwin);
    gw->dw[d]->bgd_color = lowdwin.fbx_intcolor;
    break;
  case RAISED_DWIN:
    FramedBox(gw,
	      gw->dw[d]->x0, gw->dw[d]->y0,
	      gw->dw[d]->x0 + gw->dw[d]->xl,
	      gw->dw[d]->y0 + gw->dw[d]->yl, 2, &raisedwin);
    gw->dw[d]->bgd_color = raisedwin.fbx_intcolor;
    break;
  case PLANE_DWIN:
  default:
    gw->dw[d]->bgd_color = planedwin.fbx_intcolor;
    break;
  }
}

void setclipdwin( GWIN *gw, short d ) {
  /* Set the clip rectangle to the borders of the current dwin */
  XRectangle rectangle[1];

  if( gw->clipped_dwin == d ) return;
  if( gw->clipped_dwin != -1 ) 
    printf( "Internal Warning: Changing clip mask from %d to %d.\n"
		, gw->clipped_dwin, d ); 
  gw->clipped_dwin = d;
  rectangle[0].x=gw->dw[d]->x0;
  rectangle[0].y=gw->dw[d]->y0;
  rectangle[0].width=gw->dw[d]->xl;
  rectangle[0].height=gw->dw[d]->yl;
  XSetClipRectangles( gw->gwid, gw->gwigc
                       , 0, 0, rectangle, 1, Unsorted );
}

void resetclipdwin( GWIN *gw ) {

#if 1
  XRectangle rectangle[1];

  rectangle[0].x=0;
  rectangle[0].y=0;
  rectangle[0].width = maxgwinxsize;
  rectangle[0].height = maxgwinysize;
  XSetClipRectangles( gw->gwid, gw->gwigc
                      , 0, 0, rectangle, 1, Unsorted );
#else
  XSetClipMask( gw->gwid,gw->gwigc, None );
#endif
  gw->clipped_dwin = -1;
}

void cleardwin(GWIN *gw, short d, short style) {
  gw->dw[d]->style = style;
  XClearArea(gw->gwid, gw->gwiw, gw->dw[d]->x0, gw->dw[d]->y0, 
	     gw->dw[d]->xl, gw->dw[d]->yl, False);
  drawborderdwin( gw, d );
  _dwin_clear(gw->dw[d]);
}

short wi_clear_dwin(dwin d, short style) {
  cleardwin(wi_crt_gwin_ptr(), d, PLANE_DWIN);  
}


/*--
  \end{code}

  \funparagraph{wi\_redraw_area}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

void redwin(GWIN *gw, short d, short x0, short y0, short xl, short yl);

short wi_redraw_area(dwin d, pixed x0, pixed y0, pixed xl, pixed yl)

{
  redwin(wi_crt_gwin_ptr(), d, x0, y0, xl, yl);
  return 0;
}


/*--
  \end{code}

  \section{Local paint functions}

  \ImplementationNotes{
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/


/* forward declaration of the function that keeps everything together */
static void paintgrel(GWIN * gw, DWIN * dw, struct _grel * g );

void paintbutton(GWIN *gw, short x0, short y0,
			short x1, short y1, char *text, int mode) {
  XClearArea(gw->gwid, gw->gwiw, x0, y0, x1-x0, y1-y0, False);
  if(!mode) {
    FramedBox(gw, x0, y0, x1, y1, 2, &butup);
  } else {
    FramedBox(gw, x0, y0, x1, y1, 2, &butdown);
  }  
  XDrawString(gw->gwid, gw->gwiw, gw->gwigc, x0+3, y0+12
	      , text, strlen(text));
}

#define SWXSZ 10 /* external size of switch button */
#define SWISZ 7  /* internal size of switch button */

void paintswitch(GWIN *gw, short x0, short y0, short x1, short y1,
		 char *text, int mode)  {
  /* mode is 1 if the switch must be painted as selected */

  /* Make the space clean */
  XClearArea(gw->gwid, gw->gwiw, x0, y0, x1-x0, y1-y0, False);
  
  /* draw the border of the switch */
  if(mode)
    FramedBox(gw, x0, y0, x0+SWISZ, y0+SWISZ, 1, &switchyes);
  else
    FramedBox(gw, x0, y0, x0+SWISZ, y0+SWISZ, 1, &switchno);

  XDrawString(gw->gwid, gw->gwiw, gw->gwigc, x0+10+SWXSZ,
	      y0+SWXSZ+3, text, strlen(text));
}

void paintpoint(GWIN *gw, short x0, short y0, short col) {
/* static char styletab[2] = {1, 1}; */

/*
if(style==1)
  {
  XSetDashes(gw->gwid, gw->gwigc, 0, styletab, 2);
  XSetLineAttributes(gw->gwid, gw->gwigc, width, LineOnOffDash,
    CapButt, JoinMiter);
  }
else
  XSetLineAttributes(gw->gwid, gw->gwigc, width, LineSolid,
    CapButt, JoinMiter);
*/

XDrawPoint(gw->gwid, gw->gwiw, gw->gwigc, x0, y0);
}



/*--
\end{code}

\subsection{paintline} 
% The name of the package is expected to be prezent,
% normally in the proper ``section'' specification above.

\PackagePurpose{  % be short
}
\FunctionInterraction{ 
% The order in which functions are expected to be called
% The way to use the package in general
}
\Exceptions{
}
\GlobalEffects{
% Global effects on exported or external variables
}
\Preconditions{
% Conditions to be met when the package starts to be used
}
\Performance{
% How it is, not why and what to do
}

\begin{code}
--*/


void paintline(GWIN *gw, short x0, short y0, short x1, short y1,
  short col, short width, short style, short cap /* ignored for the moment */)

{
static char styletab[2] = {1, 1};

/*
if(style==1)
  {
  XSetDashes(gw->gwid, gw->gwigc, 0, styletab, 2);
  XSetLineAttributes(gw->gwid, gw->gwigc, width, LineOnOffDash,
    CapButt, JoinMiter);
  }
else
  XSetLineAttributes(gw->gwid, gw->gwigc, width, LineSolid,
    CapButt, JoinMiter);
*/

XDrawLine(gw->gwid, gw->gwiw, gw->gwigc, x0, y0, x1, y1);
}


/*--
\end{code}

\funparagraph{painttextpad}
% The name of the function is expected to be prezent
% in the sectioning element above

\PurposeReturn{}
\Exceptions{}
\GlobalEffects{}
\Preconditions{}
\Performance{}

\begin{code}
--*/

void painttextpad(GWIN *gw, short x0, short y0, 
    short x1, short y1, struct _tp_ext * ext) {
  short s, i;
  short nc, nl;			/* textpad dimensions in characters */
  
  nc = (x1 - x0) / font_x_size;
  nl = (y1 -y0) / font_y_size;
  FramedBox( gw, x0, y0, x1+font_x_size+7, y1+5
	    , 2, &textpd);

  {
    int first_line;		/* first line to be displayed */
    int last_line;		/* last line to be displayed */
    int line;			/* current line in the buffer */
    int y;			/* vertical screen position */

    /* select the first and last lines to be displayed */
    first_line = 0; 
    if( _tp_cursor_line(ext) >= nl ) first_line = _tp_cursor_line(ext)-nl+1;
    last_line = first_line+nl;
    if( last_line > ext->line_nb ) last_line = ext->line_nb;
    
    /* display the selected lines */
    for(  y = y0+2, line = first_line
	; line<last_line
	; y += font_y_size, line++ ) {
      XDrawString(gw->gwid, gw->gwiw, gw->gwigc, x0, y+font_y_size
		  , _tp_access_line(ext, line), _tp_line_length(ext, line));
    }

    if( _tp_cursor_row(ext) > -1 ) {
      XDrawString( gw->gwid, gw->gwiw, gw->gwigc
		  , x0+_tp_cursor_row(ext)*font_x_size
		  , y0+3+(_tp_cursor_line(ext) - first_line + 1) * font_y_size
		  , "_", 1);
    }
  }
}


/*--
\end{code}

\funparagraph{editextpad}
% The name of the function is expected to be prezent
% in the sectioning element above

\PurposeReturn{  % be short
}
\Exceptions{
}
\GlobalEffects{
% Global effects on exported or external variables
}
\Preconditions{
% Conditions to be met when the function is called
}
\Performance{
% How it is, not why and what to do about it
}

\begin{code}
--*/
	

void editextpad(GWIN *gw, DWIN * dw, struct _grel *g) {
XEvent theevent;
XKeyEvent keyevent;
KeySym thekey;
char thebuffer[12];
XMappingEvent theemap;
short finished;

/* The cursor will be represented here */
/* It will be placed at position 0 when starting the edit */

short keycount;

finished = 0;
_tp_select( g->ext, g->x1 - g->x0 );

do {
  paintgrel( gw, dw, g );
  /* wait for a key */
  XNextEvent(gw->gwid, &theevent);
  switch(theevent.type) {
  case Expose:
    if(theevent.xexpose.count == 0) {
      paintgrel(gw, dw, g);
    }
    break;
  case MappingNotify:
    XRefreshKeyboardMapping(&theemap);
    break;
  case ButtonPress:
    { short xp, yp;
      struct selection other;
      xp = theevent.xbutton.x;
      yp = theevent.xbutton.y;
      get_position_selection(&other, gw, xp, yp);
      if( ( other.hitgrel != NULL ) && ( g != other.hitgrel )) {
	finished = 1;
	_tp_process_key( g->ext, XK_Return, g->x1 - g->x0 );
	XPutBackEvent( gw->gwid, &theevent );
      }
    };
    break;
  case KeyPress:
    keycount = XLookupString (&(theevent.xkey), thebuffer, 8,
			      &thekey, 0);
    finished=_tp_process_key(g->ext, thekey ,g->x1 - g->x0);
    break;  
  default:
    break;
  }
} while (!finished);
/* clear the textpad before returning */
paintgrel( gw, dw, g);

}


void painttext(GWIN *gw, short x0, short y0, char *text) {
  XDrawString(gw->gwid, gw->gwiw, gw->gwigc, x0,
	      y0+font_y_size, text, strlen(text));
}

/*--
  \end{code}

  \funparagraph{wi\_button}  % The name of the function(s)
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

static short last_button_grel = -1;

int defbutton(GWIN *gw, short d, short x0, short y0, short xl
	      , short yl, char *text, short key) {

  setclipdwin( gw, d);
  last_button_grel = defintbutton(gw->dw[d], x0, y0, xl, yl, text, key);
  x0+=gw->dw[d]->x0;
  y0+=gw->dw[d]->y0;
  paintbutton(gw, x0, y0 , x0 + xl, y0 + yl , text, 0);
  resetclipdwin( gw );
  return last_button_grel;
}

ugrel wi_button(dwin d, pixed x0, pixed y0, pixed xl, pixed yl,
	        char *t, wi_keycode key)

{
  GWIN *g;

  last_button_grel = defbutton(g =wi_crt_gwin_ptr(), d, x0, y0, xl, yl, t, key);
  return make_ugrel(wi_get_crt_gwin(), d, last_button_grel);
}

/*--
  \end{code}

  \funparagraph{wi\_hbutton}  % The name of the function(s)
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/


int defhbutton(GWIN *gw, short d, short key) /* [FUN] */ 
     
     /* returns 1 if success, 0 if failure */
     
{
  return definthbutton(gw->dw[d], key);
}

ugrel wi_hbutton(dwin d, wi_keycode key)

{
  GWIN *g;

  last_button_grel = defhbutton(g =wi_crt_gwin_ptr(), d, key);
  return make_ugrel(g, d, last_button_grel);
}


/*--
  \end{code}

  \funparagraph{wi\_switch}  % The name of the function(s)
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

static short last_switch_grel = -1;

int defswitch(GWIN *gw, short d, short x0, short y0, short xl, short yl
	      , char *text, short *swv, short key)

/* returns 1 if success, 0 if failure */

{

  int ret;
ret = defintswitch(gw->dw[d], x0, y0, xl, yl, text, swv, key);
#ifdef PAINTNOW
x0+=gw->dw[d]->x0;
y0+=gw->dw[d]->y0;
paintswitch(gw, x0, y0, x0+xl, y0+yl, text , (*swv==key));  
#endif
return ret;
}

ugrel wi_switch(dwin d, pixed x0, pixed y0, pixed xl, pixed yl,
		char *t, wi_keycode *swv, wi_keycode key)

{
  GWIN *g;

  g =wi_crt_gwin_ptr();
  last_switch_grel = defswitch(g, d, x0, y0, xl, yl, t, swv, key);
  return make_ugrel(g, d, last_switch_grel);
}



/*--
  \end{code}

  \funparagraph{wi\_alt\_switch}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

int defaltswitch(GWIN *gw, short d, short x0, short y0, short xl, short yl,
char *text, short *swv, short key)

/* returns 1 if success, 0 if failure */

{
int ret;
ret = defintaswitch(gw->dw[d], x0, y0, xl, yl, text, swv, key);
#ifdef PAINTNOW
x0+=gw->dw[d]->x0;
y0+=gw->dw[d]->y0;
paintswitch(gw, x0, y0, x0+xl, y0+yl, text, *swv==key);
#endif
return ret;
}

ugrel wi_alt_switch(dwin d, pixed x0, pixed y0, pixed xl, pixed yl,
		char *t, wi_keycode *swv, wi_keycode key)

{
  GWIN *g;

  g =wi_crt_gwin_ptr();
  last_switch_grel = defaltswitch(g, d, x0, y0, xl, yl, t, swv, key);
  return make_ugrel(g, d, last_switch_grel);
}


/*--
  \end{code}

  \funparagraph{wi\_text}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

short last_text_grel;
int gtext(GWIN *gw, short d, short x, short y, char *text) {
  int ret;
  ret = definttext(gw->dw[d], x, y, gw->dw[d]->cr_col,text);
#ifdef PAINTNOW
  painttext(gw, gw->dw[d]->x0+x, gw->dw[d]->y0+y,text);
#endif
  return ret;
}

ugrel wi_text(dwin d, pixed x, pixed y, char *text)

{
  GWIN *g;

  g =wi_crt_gwin_ptr();
  last_text_grel = gtext(g, d, x, y, text);
  return make_ugrel(g, d, last_text_grel);
}



/*--
  \end{code}

  \funparagraph{wi\_textpad}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/


int deftextpad(GWIN *gw, short d, short x0, short y0, short nc, short nl,
  short size, char *buffer, short key) {

  return definttextpad(gw->dw[d], x0, y0, nc, nl, buffer, size, key);
}


ugrel wi_textpad(dwin d, pixed x, pixed y, pixed nc, pixed nl, short ssize,
		 char *text, wi_keycode key)

{
  GWIN *g;

  g = wi_crt_gwin_ptr();
  last_text_grel = deftextpad(g, d, x, y, nc, nl, ssize, text, key); 
  return make_ugrel(g, d, last_text_grel);
}

ugrel wi_line(dwin d, pixed x0, pixed y0, pixed x1, pixed y1)

{
  GWIN *g;

  gmoveto( g =wi_crt_gwin_ptr(), d, x0, y0);
  printf("after gmoveto\n");
  glineto(g, d, x1, y1);
  printf("after glineto\n");
  return make_ugrel(wi_get_crt_gwin(), d, last_line_grel);
}

ugrel wi_textbuffer(dwin d, pixed x, pixed y, pixed nc, pixed nl
		 , short size, char *text, wi_keycode key)

{
  GWIN *gw;

  gw = wi_crt_gwin_ptr();
  last_text_grel =
    definttextpad(gw->dw[d], x, y, nc, nl, text, size, key);
  if( last_text_grel > -1 ) { 
    cursor_hide();
    painttextpad(gw, gw->dw[d]->x0+x, gw->dw[d]->y0+y
		 , gw->dw[d]->x0+x+nc*font_x_size
		 , gw->dw[d]->y0+y+nl*font_y_size+1
		 , ndx_to_grel(gw->dw[d], last_text_grel)->ext);
    cursor_show();
  }
  return make_ugrel(gw, d, last_text_grel);
}

short wi_interact( void ) {
  return gwinteract( wi_crt_gwin_ptr() );
}

/*--
\end{code}

\funparagraph{wi\_rect}  % The name of the function(s)
\ImplementationNotes{ 
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

\begin{code}
--*/

ugrel wi_rect(dwin d, pixed x0, pixed y0, pixed xl,  pixed yl ) {
  DWIN * the_dwin;
  short the_grel;

  the_dwin = wi_crt_gwin_ptr()->dw[d];
  the_grel = defintrect( the_dwin, x0, y0, x0+xl, y0+yl, the_dwin->cr_col );
  paintgrel( wi_crt_gwin_ptr(), the_dwin, ndx_to_grel( the_dwin, the_grel ) );
  return make_ugrel(wi_crt_gwin_ptr(), d, the_grel);
}

/*--
  \end{code}

  \section{Interaction functions}

  \ImplementationNotes{
  % How implementation of the whole package is done: algorithms---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

void gsetcolor(GWIN *gw, short d, short col)

{
gw->dw[d]->cr_col = col;
}

void gsetlinestyle(GWIN *gw, short d, short mask)

{
gw->dw[d]->cr_style = mask;
}

void gsetlinewidth(GWIN *gw, short d, short width)

{
gw->dw[d]->cr_width = width;
}

void gsetlinecap(GWIN *gw, short d, short cap)

{
gw->dw[d]->cr_cap = cap;
}


void gmoveto(GWIN *gw, short d, short x, short y)

{
gw->dw[d]->cr_draw_x = x;
gw->dw[d]->cr_draw_y = y;
}

void glineto(GWIN *gw, short d, short x, short y)

{
#ifdef PAINTNOW
setclipdwin( gw, d );
paintline(gw, 
  gw->dw[d]->x0+gw->dw[d]->cr_draw_x,
  gw->dw[d]->y0+gw->dw[d]->cr_draw_y,
  gw->dw[d]->x0+x, gw->dw[d]->y0+y, gw->dw[d]->cr_col,
  gw->dw[d]->cr_width, gw->dw[d]->cr_style, gw->dw[d]->cr_cap);
resetclipdwin( gw );
#endif
last_line_grel = defintline(gw->dw[d], gw->dw[d]->cr_draw_x, gw->dw[d]->cr_draw_y,
			    x, y, gw->dw[d]->cr_col,  gw->dw[d]->cr_width, gw->dw[d]->cr_style, 
			    gw->dw[d]->cr_cap);
gw->dw[d]->cr_draw_x = x;
gw->dw[d]->cr_draw_y = y;
return ;
}


void gpoint(GWIN *gw, short d, short x, short y)

{

#ifdef PAINTNOW
setclipdwin( gw, d );
paintpoint(gw, 
  gw->dw[d]->x0+x,
  gw->dw[d]->y0+y, gw->dw[d]->cr_col);
resetclipdwin( gw );
#endif
defintpoint(gw->dw[d], x, y, gw->dw[d]->cr_col);
return ;
}


static void paintgrel(GWIN * gw, DWIN * dw, struct _grel * g )  { /* [FUN] */
  /* paints a grel on the screen */
  short nb, * ext;
  short lastx, lasty;
  
  switch(g->gtype) {
  case POINTGREL:
/*  case MULTIPOINTGREL:*/
/*    _setcolor(dw -> cr_col);*/
    paintpoint(gw, dw -> x0 + g -> x0, dw -> y0 + g -> y0, dw -> cr_col);
    ext = g -> ext;
    for (nb = g -> extfill; nb; nb -= 2, ext += 2) {
      paintpoint(gw, ext[0], ext[1], dw -> cr_col);
    }
    break;
  case BUTTONGREL:
    paintbutton(gw, dw->x0 + g->x0, dw->y0 + g->y0,
                dw->x0 + g->x1, dw->y0 + g->y1, g->ext, g->swvval);
    break;
  case SWITCHGREL:
  case ASWITCHGREL:
    paintswitch(gw, dw->x0 + g->x0, dw->y0 + g->y0,
                dw->x0 + g->x1, dw->y0 + g->y1, g->ext,
                g->gkey==g->swvval);
    break;
  case LINEGREL:
/*  case MULTILINEGREL:*/
    paintline(gw, dw->x0 + g->x0, dw->y0 + g->y0,
              dw->x0 + g->x1, dw->y0 + g->y1,
              g->gcolor, g->gsize, g->gstyle, g->gcap);
    lastx = dw->x0 + g->x1;
    lasty = dw->y0 + g->y1;
    ext = g -> ext;
    for (nb = g -> extfill; nb; nb -= 2, ext += 2) {
      XDrawLine(gw->gwid, gw->gwiw, gw->gwigc, lastx, lasty, ext[0], ext[1]);
      lastx = ext[0];
      lasty = ext[1];
    }
    break;
  case RECTGREL:
    XSetForeground( gw->gwid, gw->gwigc, g->gcolor);
    XFillRectangle( gw->gwid, gw->gwiw, gw->gwigc
		   , dw->x0 + g->x0, dw->y0 + g->y0
		   , g->x1-g->x0, g->y1-g->y0 );
    break;
  case TEXTGREL:
    painttext(gw, dw->x0 + g->x0,
              dw->y0 + g->y0, g->ext);
    break;
  case TEXTPADGREL:
    painttextpad(gw, dw->x0 + g->x0, dw->y0 + g->y0
		 , dw->x0 + g -> x1, dw->y0 + g -> y1
		 , g->ext );
    break;
  case HBUTTONGREL:
  default:
    break;
  }     
}

static void paintgrel_active(GWIN * gw, DWIN * dw, struct _grel * g )  { /* [FUN] */
  /* paints a grel on the screen */
  short nb, * ext;
  short lastx, lasty;
  
  switch(g->gtype) {
  case POINTGREL:
/*  case MULTIPOINTGREL:*/
/*    _setcolor(dw -> cr_col);*/
    paintpoint(gw, dw -> x0 + g -> x0, dw -> y0 + g -> y0, dw -> cr_col);
    ext = g -> ext;
    for (nb = g -> extfill; nb; nb -= 2, ext += 2) {
      paintpoint(gw, ext[0], ext[1], dw -> cr_col);
    }
    break;
  case BUTTONGREL:
    paintbutton(gw, dw->x0 + g->x0, dw->y0 + g->y0,
                dw->x0 + g->x1, dw->y0 + g->y1, g->ext, 0);
    break;
  case SWITCHGREL:
  case ASWITCHGREL:
    paintswitch(gw, dw->x0 + g->x0, dw->y0 + g->y0,
                dw->x0 + g->x1, dw->y0 + g->y1, g->ext,
                g->gkey==g->swvval);
    break;
  case LINEGREL:
/*  case MULTILINEGREL:*/
    paintline(gw, dw->x0 + g->x0, dw->y0 + g->y0,
              dw->x0 + g->x1, dw->y0 + g->y1,
              g->gcolor, g->gsize, g->gstyle, g->gcap);
    lastx = dw->x0 + g->x1;
    lasty = dw->y0 + g->y1;
    ext = g -> ext;
    for (nb = g -> extfill; nb; nb -= 2, ext += 2) {
      XDrawLine(gw->gwid, gw->gwiw, gw->gwigc, lastx, lasty, ext[0], ext[1]);
      lastx = ext[0];
      lasty = ext[1];
    }
    break;
  case TEXTGREL:
    painttext(gw, dw->x0 + g->x0,
              dw->y0 + g->y0, g->ext);
    break;
  case TEXTPADGREL:
    painttextpad(gw, dw->x0 + g->x0, dw->y0 + g->y0
		 , dw->x0 + g -> x1, dw->y0 + g -> y1
		 , g->ext );
    break;
  case HBUTTONGREL:
  default:
    break;
  }     
}


void redwin(GWIN *gw, short d, short x0, short y0, short xl, short yl) /* [FUN] */ 
{
  struct _grel *g;
  DWIN * dw;
  short i, j;
  
#define GREL_IN_WINDOW(g) ((g->x0 >= x0 && g->x0 < xl && g->y0 >= y0 \
                            && g->y0 < yl) || \
                           (g->x1 >= x0 && g->x1 < xl && g->y1 >= y0 \
                            && g->y1 < yl))
  setclipdwin( gw, d );
  dw = gw->dw[d];
  if(x0==0 && y0==0 && xl==0 && yl==0)
    {
      xl = dw->xl;
      yl = dw->yl;
    }
  /* for all the full pages */
  for(i=0; i<dw->gpfill - 1 ; i++) {
    for(j=0; j< NOGPERPAGE; j++) {
      g = dw->gp[i] + j;
      if (GREL_IN_WINDOW(g))
        paintgrel(gw, dw, g);
    } 
  }
  /* for the last page */
  i = dw-> gpfill - 1;
  for(j=0; j<gw->dw[d]->gppfill; j++)
    {
      g = dw->gp[i] + j;
      if (GREL_IN_WINDOW(g))
        paintgrel(gw, dw, g);
    }
  resetclipdwin( gw );
}

short spotshotx, spotshoty;

void setspotshot(GWIN *gw, short d, short key)

{
gw->dw[d]->spotshotkey = key;
}

void sethorshot(GWIN *gw, short d, short key)

{
gw->dw[d]->horshotkey = key;
}

void setvershot(GWIN *gw, short d, short key)

{
gw->dw[d]->vershotkey = key;
}

void setrectshot(GWIN *gw, short d, short key)

{
gw->dw[d]->rectshotkey = key;
}


/********************* The inquiry functions **********************************/

int dwin_x_size(GWIN *g, short d) {
  return g->dw[d]->xl;
}

int dwin_y_size(GWIN *g, short d) {
  return g->dw[d]->yl;
}


/********************* The interaction functions ******************************/

short gwinpeep(GWIN *gw)

{
return XPending(gw->gwid);
}


void reverseswitch(GWIN *gw, DWIN *dw, struct _grel *g)
  
{
if(g->swvval==g->gkey)
  g->swvval = 0;
else
  g->swvval = g->gkey;
paintswitch(gw, dw->x0+g->x0, dw->y0+g->y0, g->x1-g->x0,
  g->y1-g->y0, g->ext, (g->swvval==g->gkey));
return ;
}

void pressaswitch(GWIN *gw, DWIN *dw, struct _grel *g)

{
short j, k, ng;
struct _grel *reg;

/* This function makes a change only if the aswitch was
 * not already pressed
 */
if(g->swvval==g->gkey)
  return;

/* set the switch to ``pressed'' */
g->swvval=g->gkey;
/* paint the switch ``pressed'' */
paintswitch(gw, dw->x0+g->x0, dw->y0+g->y0, g->x1-g->x0,
  g->y1-g->y0, g->ext, 1);

/* find the other ASWITCH having the same gkey and being pressed
 * --- if there is any one 
 */

  for(j=0; j<dw->gpfill; j++)
    {
    reg = dw->gp[j];
    ng = NOGPERPAGE;
    if(j==dw->gpfill-1)
      ng = dw->gppfill;
    for(k = 0; k<ng; k++)
      {
      if(reg!=g && reg->gtype==ASWITCHGREL && 
         reg->gswv==g->gswv) /* && reg->swvval==reg->gkey)*/
        {
        reg->swvval=g->gkey;
        paintswitch(gw, dw->x0+reg->x0, dw->y0+reg->y0, reg->x1-reg->x0,
          reg->y1-reg->y0, reg->ext, 0);
        /* same is done for all of them as it results */
        }
      reg++;
      }
    }
}



void gset_tp_contents(GWIN *gw, short d, short key, char *cts) {
  struct _grel *g;
  struct selection sel;
  
  get_key_selection(&sel, gw, key);
  g = sel.hitgrel;
  if(g==NULL)
    return;  /* no such key */
  if(g->gtype!=TEXTPADGREL)
    return;  /* grel is not a textpad */
  if(g->ext==NULL)
    return;  /* grel is badly allocated */
  _tp_contents( g->ext, cts, g->x1-g->x0 );
  paintgrel( gw, sel.lasthitdwin, g);
}

enter_graphic_block()  

     {
     }

exit_graphic_block()

     {
     }


get_screen_colors(short *r, short *g, short *b)

{
  short i;
  long a;
/*
  for(i=0; i<NOFSCOLORS; i++) {
    a = scol_red[i];
    a *= 10000;
    a /= 256;
    r[i] = a;
    a = scol_green[i];
    a *= 10000;
    a /= 256;
    g[i] = a;
    a = scol_blue[i];
    a *= 10000;
    a /= 256;
    b[i] = a;
  }
 */
}

   static struct _grel * toggle_key(struct _grel * g) { /* [FUN] */ 
  if (g -> swvval)
    g -> swvval = 0;
  else
    g -> swvval = g -> gkey;
  return g;
}


   
short gwinteract(GWIN *gw) {
  XEvent theevent;
  short xp, yp;
  struct _grel *g;
  char *previous_help;
  struct selection sel;
  short i;

  previous_help = NULL;
while(1)  /* forever --- before something happens to provoke the return */
  {
    XNextEvent(gw->gwid, &theevent);
  switch(theevent.type)
    {
    case Expose:
      if(theevent.xexpose.count == 0) {
        for(i=0; i<gw->dwinfill; i++) {
#if 0
	  XClearArea(gw->gwid, gw->gwiw, gw->dw[i]->x0, gw->dw[i]->y0,
		     gw->dw[i]->xl
                   , gw->dw[i]->yl, False);
#endif
	  drawborderdwin( gw, i );
          redwin(gw, i, 0, 0, 0, 0);
	}
      }
      break;
    case MotionNotify:
      break;
      if( gw->help_dwin == -1 ) break; /* no help dwin set */
      xp = theevent.xmotion.x;
      yp = theevent.xmotion.y;
      if( gw->help_on && 0 ) {
	char * help_string;
	printf("helping\n");
	help_string = _locate_help( gw, xp, yp );
	if( previous_help != help_string ) {
	  wi_clear_dwin( gw->help_dwin, 0 );
	  previous_help = help_string;
	}
	if( help_string != NULL )
	  gtext( gw, gw->help_dwin, 3, 3, help_string);
      }
      break;
    case ButtonPress:
      xp = theevent.xbutton.x;
      yp = theevent.xbutton.y;
      get_position_selection(&sel, gw, xp, yp);
      g = sel.hitgrel;
      if(g==NULL) {
	short d;

	for(d=gw->dwinfill-1; d>=0; d--) {
          if(gw->dw[d]->spotshotkey &&
             xp >= gw->dw[d]->x0 && xp < gw->dw[d]->x0+gw->dw[d]->xl &&
             yp >= gw->dw[d]->y0 && yp < gw->dw[d]->y0+gw->dw[d]->yl) {
	    xspotshot = xp - gw->dw[d]->x0;
	    yspotshot = yp - gw->dw[d]->y0;
	    return( gw->dw[d]->spotshotkey );
	  }
	}
      }
      switch(g->gtype)
        {
        case BUTTONGREL:
	   if( theevent.type == ButtonPress )  {
	     paintgrel(gw, sel.lasthitdwin, toggle_key(g));
	     do {
	       XNextEvent( gw->gwid, &theevent );
	     } while( theevent.type != ButtonRelease );
	     paintgrel(gw, sel.lasthitdwin, toggle_key(g));
	   }
	  return g->gkey;
        case SWITCHGREL:
          reverseswitch(gw, sel.lasthitdwin, g);  
            /* changes it and makes the display */
          break;
        case ASWITCHGREL:
          pressaswitch(gw, sel.lasthitdwin, g);
          break;
        case TEXTPADGREL:
          editextpad(gw, sel.lasthitdwin, g);
	  return( g->gkey );
          break;  
        default:
          break; 
        }
      break;
    case MappingNotify:
      printf("mappingnotify\n");
/*
      XRefreshKeyboardMapping(&theevent);
*/
      break;
    case KeyPress:  /* WHAT'S THIS? */
                    /* IGNORE FOR THE MOMENT */
      /*
      keycount = XLookupString (&theevent, thebuffer, 8,
				&thekey, 0);
       if((keycount == 1) && (thebuffer[0] == 'q'))
	 finished = TRUE;
      */
      break;
    }
  }
}

/*--
  \end{code}

  \funparagraph{wi\_accept}  % The name of the function(s)
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

void gwaccept(GWIN * gw) { /* [FUN] */ 
  /* copy the internal values of the switches in the user's variables */
  ITERATOR it;
  short d;
  struct _grel * temp;
  
  /* map over all the dwins of the gwin */
/*  printf("CALL gwaccept\n");*/
  for (d = 0; d < gw -> dwinfill; d ++) {
    init_bwd_grel_iterator(gw, d, &it);
    while (check_lower_bound_grel(&it)) {
      temp = access_grel(&it);
      switch (temp -> gtype) {
      case SWITCHGREL:
      case ASWITCHGREL:
        * (temp -> gswv) = temp -> swvval;
        break;
      case TEXTPADGREL:
        strcpy((char *)(temp->gswv), _tp_access_contents(temp->ext));
        break;
      default:
        break;
      }
      iterator_backward(&it);
    }
  }
}

wi_accept()

{
  GWIN *g;

  g = wi_crt_gwin_ptr();
  gwaccept(g);
}

void wi_inchg_text_val(dwin d, wi_keycode key, char *cts)

{
  GWIN *g;

  g = wi_crt_gwin_ptr();
  gset_tp_contents(g, d, key, cts);
}

int gwipeep() { /* [FUN] */ 
  return 1;  /* don't work yet */
}

/*--
\end{code}

% MOVE THIS COMMENT TO THE END OF THE FILE
\endoffile

--*/
