/*    WIPort 2 RESources implementation
 *
 *    Copyright (c) 1994--1998 Alexandru Dan Corlan
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU Library General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    In short, you can use and modify this program. You can
 *    distribute the program with or without your modifications if you want,
 *    but you must not require licence fees and must provide the
 *    full source code with the distribution. If you make an application program
 *    which just calls functions from this library you may distribute that
 *    application program without source code, but also as an linkable object file, 
 *    and must distribute also this library with source code.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Library General Public License for more details.
 *
 *    You should have received a copy of the GNU Library General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



#include <stdio.h>
#include <malloc.h>
#include "wipres.h"

/*--

\Definition{ % What the data structure is designed to represent
}
\SemanticStructure{
% Description of structure and associated semantics
% Full definition of the intented data types by means of ADA declarations
% may be appropiate
}
\ConsistencyRules{
% Rules that the data in the structure is expected to obey in order
% to be legal, even if not enforced by the program itself
}

--*/

#define FLAG_RES 0
#define INT_RES  1
#define STR_RES  2
#define FLO_RES  3
#define CELL_RES 4

struct res {
  char *name;
  char *cmt;
  short type;
  union {
    float flo_;
    long int_;
    char *str_;
  } val;
};

struct res theres[MAXRES];
short theres_fill = 0;


char finname[128] = "";
char crtprefix[128] = "";


/*--

\ImplementationNotes{
% How implementation of the function is done: algorithm---if special,
% implementation decisions, discarded implementation alternatives
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

--*/

void wipres()

{
  theres_fill = 0;
}

/*--

% The name of the function is expected to already be prezent
\PurposeReturn{  % be short
The folowing function locates a resource by name---and returns its index;
it returns -1 if the resource could not be located sucessfully.
}
\Exceptions{
}
\GlobalEffects{
% Global effects on exported or external variables
}
\Preconditions{
% Conditions to be met when the function is called
}
\Performance{
% How it is, not why and what to do about it
Strightforward sequential search, quite ineffective.
}

\ImplementationNotes{
% How implementation of the function is done: algorithm---if special,
% implementation decisions, discarded implementation alternatives
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
Better search if prooved necessary by future applications. This function
is the only means to locate resources by names that is used in the package.
}

--*/

static short reslocate(char *n)

{
  short i;

  for(i=0; i<theres_fill; i++)
    if(!strcmp(n, theres[i].name))
      return i;
  return -1;
}

/*--

% The name of the function is expected to already be prezent
\PurposeReturn{  % be short
}
\Exceptions{
}
\GlobalEffects{
  % Global effects on exported or external variables
}
\Preconditions{
  % Conditions to be met when the function is called
}
\Performance{
  % How it is, not why and what to do about it
}

\ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

--*/

static char *prefix_paste(char *name) {

  short i, j;

  i = 0;
  j = 0;
  for(i=j=0; crtprefix[i]>0; i++, j++)  /* SIZES SHOULD BE CHECKED */
    finname[j]=crtprefix[i];
  for(i=0; name[i]>0; i++, j++)
    finname[j] = name[i];
  finname[j] = 0;
  return finname;
}

/*--

\ImplementationNotes{
% How implementation of the function is done: algorithm---if special,
% implementation decisions, discarded implementation alternatives
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
Add true and detailed exception processing, and also test for the consistency
of the arguments.
}

--*/

short sresdef(char *name, char *str, char *cmt)

{
  short l;

  if(name!=NULL) {
    name = prefix_paste(name);
    l = reslocate(name);
    if(l==-1) {  /* defining a new resource */
      if(theres_fill>=MAXRES)
        return 0;
      theres[theres_fill].name = (char *)malloc(strlen(name)+1);
      if(theres[theres_fill].name==NULL)
        return 0;
      strcpy(theres[theres_fill].name, name);
      l = theres_fill;
      theres[theres_fill].cmt = NULL;
      theres[theres_fill].val.str_ = NULL;
    }
  }
  else {
    return 0;
  }
  
  if(cmt!=NULL) {
    if(theres[l].cmt!=NULL)
      free(theres[l].cmt);
    theres[l].cmt = (char *)malloc(strlen(cmt)+1);
    if(theres[l].cmt==NULL) {
      free(theres[l].name);
      return 0;
    }
    strcpy(theres[l].cmt, cmt);
  } else
    theres[l].cmt = NULL;


  if(str!=NULL) {
    if(theres[l].val.str_!=NULL)
      free(theres[l].val.str_);
    theres[l].val.str_ = (char *)malloc(strlen(str)+1);
    if(theres[l].val.str_==NULL) {
      free(theres[l].name);
      if(theres[l].cmt!=NULL)
        free(theres[l].cmt);
      return 0;
    }
    strcpy(theres[l].val.str_, str);
  } else
    theres[l].val.str_ = NULL;
  theres[l].type = STR_RES;
  if(theres_fill == l)
    theres_fill ++;
  return 1;
}


short iresdef(char *name, long argint, char *cmt)

{
  short l;
  
  if(name!=NULL) {
    name = prefix_paste(name);
    l = reslocate(name);
    if(l==-1) { /* defining a new resource for the name */
      if(theres_fill>=MAXRES)
        return 0;
      theres[theres_fill].name = (char *)malloc(strlen(name)+1);
      if(theres[theres_fill].name==NULL)
        return 0;
      strcpy(theres[theres_fill].name, name);
      l = theres_fill;
      theres[l].cmt = NULL;
    }
  }
  else {
    return 0;
  }
  
  if(cmt!=NULL) {
    theres[l].cmt = (char *)malloc(strlen(cmt)+1);
    if(theres[l].cmt==NULL) {
      free(theres[l].name);
      return 0;
    }
    strcpy(theres[l].cmt, cmt);
  } else
    theres[l].cmt = NULL;

  theres[l].type = INT_RES;
  theres[l].val.int_ = argint;
  if(l==theres_fill)
    theres_fill ++;
  return 1;
}


short fresdef(char *name, float argflo, char *cmt)
     
{
  short l;

  if(name!=NULL) {
    name = prefix_paste(name);
    l = reslocate(name);  
    if(l==-1) {
      if(theres_fill>=MAXRES)
        return 0;
      theres[theres_fill].name = (char *)malloc(strlen(name)+1);
      if(theres[theres_fill].name==NULL)
        return 0;
      strcpy(theres[theres_fill].name, name);
      theres[theres_fill].cmt = NULL;
      l = theres_fill;
    }
    else {
      return 0;
    }
    
    if(cmt!=NULL) {
      if(theres[l].cmt!=NULL)
        free(theres[l].cmt);
      theres[l].cmt = (char *)malloc(strlen(cmt)+1);
      if(theres[l].cmt==NULL) {
        free(theres[l].name);
        return 0;
      }
      strcpy(theres[l].cmt, cmt);
    } else
      theres[l].cmt = NULL;
  }
  
  theres[l].type = FLO_RES;
  theres[l].val.flo_ = argflo;
  if(theres_fill==l)
    theres_fill++;
  return 1;
}

/*--

\ImplementationNotes{
% How implementation of the function is done: algorithm---if special,
% implementation decisions, discarded implementation alternatives
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

--*/


char *sres(char *name)

{
  short i;

  if(name==NULL)
    return NULL;
  name = prefix_paste(name);
  i = reslocate(name);
  if(i>-1)
    return theres[i].val.str_;
  return NULL;
}


long ires(char *name)

{
  short i;

  if(name==NULL)
    return 0;
  name = prefix_paste(name);
  i = reslocate(name);
  if(i>-1)
    return theres[i].val.int_;
  return 0;
}



float fres(char *name)

{
  short i;

  if(name==NULL)
    return 0.0;
  name = prefix_paste(name);
  i = reslocate(name);
  if(i>-1)
    return theres[i].val.flo_;
  return 0.0;
}

/*--

\ImplementationNotes{
% How implementation of the function is done: algorithm---if special,
% implementation decisions, discarded implementation alternatives
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

--*/

/* Internal exceptions such as a wrong type are simply ignored */
/* They return 0 if the name is not found and 1 if found---value is set */

short sresset(char *name, char *str)

{
  short i;
  char *newp;

  if(name==NULL)
    return 0;
  name = prefix_paste(name);
  i = reslocate(name);
  if(i>-1) {
    newp = (char *)malloc(strlen(str)+1);
    if(newp==NULL)
      return 0;
    if(theres[i].val.str_!=NULL)
      free(theres[i].val.str_);
    theres[i].val.str_ = newp;
    return 1;
  }
  return 0;
}


short iresset(char *name, long iv)

{
  short i;

  if(name==NULL)
    return 0;
  name = prefix_paste(name);
  i = reslocate(name);
  if(i>-1) {
    theres[i].val.int_ = iv;
    return 1;
  }
  return 0;
}



short fresset(char *name, float fv)

{
  short i;

  if(name==NULL)
    return 0;
  name = prefix_paste(name);
  i = reslocate(name);
  if(i>-1) {
    theres[i].val.flo_ = fv;
    return 1;
  }
  return 0;
}


/*--

\ImplementationNotes{
% How implementation of the function is done: algorithm---if special,
% implementation decisions, discarded implementation alternatives

The resread function does not initialize the table so that multiple
resource files can be combine; if the name of a resource is encountered
twice it comes that the value of the first resource is replaced
with the value of the second resource.

}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

--*/

short reswrite(char *fnw) {

  FILE *fw;
  short i, j, c;

  setresprefix("");
  fw = fopen(fnw, "w");
  if(fw==NULL)
    return 0;
  for(i=0; i<theres_fill; i++) {
    fprintf(fw, "%-64s", theres[i].name);  /* pardon the magic */
    switch(theres[i].type) {
    case STR_RES:
      fprintf(fw, "S \"");
      for(j=0; c = theres[i].val.str_[j]; j++)
        switch(c) {
        case '"':
          putc('\\', fw);
          putc('"', fw);
          break;
        default:
          putc(c, fw);
          break;
        }
      putc('"', fw);
      putc('\n', fw);
      break;
    case INT_RES:
      fprintf(fw, "I %ld\n", theres[i].val.int_);
      break;
    case FLO_RES:
      fprintf(fw, "F %f\n", theres[i].val.flo_);
      break;
    default:
      break;  /* type is unknown, resource ignored */
    }
  }
  fclose(fw);
  return 1;
}

short resread(char *fnr) {

  FILE *fr;
  short i, t, c;
  short l;
  static char frb[80];  /* pardon the magic --- this will be the name  */
  static char frv[256]; /* pardon the magic --- this will be the value */

  setresprefix("");  
  fr = fopen(fnr, "r");
  if(fr==NULL)
    return 0;
  while((c=getc(fr))!=EOF) {
    frb[0] = c;  /* read and trim the name of the resource */
    for(i=1; i<64; i++)
      if( (frb[i] = getc(fr) ) == ' ' ) break;
    frb[i] = 0;
    do {
      t = getc(fr);  /* this must be the type */
    } while( t == ' ' );
    getc(fr);      /* this is the space */
    switch(t) {
    case 'S':
      getc(fr);    /* " */
      for(i=0; i<256; i++) {
        c = getc(fr);
        if(c=='"' || c==EOF)
          break;
        frv[i] = c;
      }
      frv[i] = 0;
      while(c!='\n' && c!=EOF)  /* Comment loading should be inserted here */
        c = getc(fr);
      if(!sresset(frb, frv))
        if(!sresdef(frb, frv, NULL))
          return 0;
      break;
    case 'I':
      i = 0;
      while((c=getc(fr))!='\n' && c!=EOF && i<255)
        frv[i++] = c;
      frv[i] = 0;
      if(!iresset(frb, atol(frv)))
        if(!iresdef(frb, atol(frv), NULL))
          return 0;
      break;
    case 'F':
      i = 0;
      while((c=getc(fr))!='\n' && c!=EOF && i<255)
        frv[i++] = c;
      frv[i] = 0;
      if(!fresset(frb, atof(frv)))
        if(!fresdef(frb, atof(frv), NULL))
          return 0;
      break;
    default:
      break;
    }
  }
  fclose(fr);
  return 1;
}

/*--
\ImplementationNotes{
% How implementation of the function is done: algorithm---if special,
% implementation decisions, discarded implementation alternatives
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}
--*/

short setresprefix(char *pf)

{
  if(pf==NULL)
    crtprefix[0] = 0;
  else
    strcpy(crtprefix, pf);
  return 1;  /* OK always */
}


/*--
  \end{code}

  % MOVE THIS COMMENT TO THE END OF THE FILE
  \endoffile

  --*/

