/*    WIPTIME implementation of time input via WIPORT
 *
 *    Copyright (c) 1994--1998 Alexandru Dan Corlan, MD
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU Library General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    In short, you can use and modify this program. You can
 *    distribute the program with or without your modifications if you want,
 *    but you must not require licence fees and must provide the
 *    full source code with the distribution. If you make an application program
 *    which just calls functions from this library you may distribute that
 *    application program without source code, but also as an linkable object file, 
 *    and must distribute also this library with source code.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Library General Public License for more details.
 *
 *    You should have received a copy of the GNU Library General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

/*--

  \FileTitle{ % format: <short title> --- <long title>
  WIPTIME --- Time input/output in WIPORT2.
  }
  \CopyrightNotice{ % format: Copyright \copyright <year(s)> <who>
  Copyright \copyright 1994 Alexandru Dan CORLAN
  }
  \FileAuthor{ % the full name of the author
  Alexandru Dan CORLAN
  }
  \AuthorAffiliation{ % Intitution of the author when writing the file
  Electrocardiognosis SRL, 28 Argentina str, 71206, Bucharest 1, ROMANIA
  }
  \RevisionHistory{ % write freely the revision history here
  $Log: wiptime.c,v $
 * Revision 1.2  1996/02/25  22:38:01  dcorlan
 * The wiport2l system, which now compiles from the box for MSDOS (I presume)
 * but anyway with svgalib-1.2.8  and grx-2.0, and with X11R6. Nevertheless,
 * this is the version shipped by mseritan and it does not work well.
 * Something is fishy with the keycodes and the mouse enters a dubious mode,
 * and it works veery slow and the current widget flickers on the screen.
 * It sometimes crashes when pressing a common button.
 *
 * wiport2x.c is actually obsolete.
 *
 * What I fear is that the files in the CVS repository are not the files
 * we actually use in the compilation.
 *
 * New things:
 *   wiport.fnt, wiport.fna are the new fonts.
 *   I don't know what should be the difference, if any, between Makefile
 *   and makefile.
 *
 * Revision 1.1  1995/01/15  19:42:23  root
 * Initial revision
 *
  }
  \vskip 1in

  \section{Wiport Time Implementation}

  \ImplementationNotes{
  This package implements interactive editing of time items, calling
  WIPORT entries. It does not depent on anything else than WIPORT2.
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

#include <stdio.h>
#include <time.h>

#include "wiport2.h"
#include "wiptime.h"

/*--
  \end{code}

  \dataparagraph{Internal time structure}

  \Definition{
  Broken down representation of time, stolen from the Unix
  {\tt <time.h>} file. It is identical to that structure,
  but reproduced here in order not to depend on {\tt <time.h>}.
  }

  \SemanticStructure{
  The following section is from the {\tt ctime} man page from
  Linux V1.0.9, Slackware 2.0.0 distribution.

  \small
  The members of the tm structure are:
  
  \begin{description}
  \item[{\tt tm\_sec}] The number of seconds after the minute, normally in
  the range 0 to 59, but can be up to 61 to allow for
  leap seconds.
  
  \item[{\tt tm\_min}] The number of minutes after the hour, in the  range
  0 to 59.

  \item[{\tt tm\_hour}] The  number  of hours past midnight, 
  in the range 0 to 23.

  \item[{\tt tm\_mday}] The day of the month, in the range 1 to 31.

  \item[{\tt tm\_mon}] The number of months since January, in the range  0
              to 11.

   \item[{\tt tm_year}] The number of years since 1900.

   \item[{\tt tm\_wday}] The  number of days since Sunday,
        in the range 0 to 6.

   \item[{\tt tm\_yday}] The number of days since January 1, in the range  0
              to 365.

   \item[{\tt tm\_isdst}]
              A  flag that indicates whether daylight saving time
              is in effect at the time described.  The  value  is
              positive if daylight saving time is in effect, zero
              if it is not, and negative if  the  information  is
              not available.
\end{description}
  }
  \ConsistencyRules{
  }

  \begin{code}
  --*/

#ifdef NOTIME
struct tm
{
  int     tm_sec;         /* seconds */
  int     tm_min;         /* minutes */
  int     tm_hour;        /* hours */
  int     tm_mday;        /* day of the month */
  int     tm_mon;         /* month */
  int     tm_year;        /* year */
  int     tm_wday;        /* day of the week */
  int     tm_yday;        /* day in the year */
  int     tm_isdst;       /* daylight saving time */
};
#endif

/*--
  \end{code}

  \dataparagraph{Time Edit Widget}

  \Definition{ % What the data structure is designed to represent
  }
  \SemanticStructure{
  % Description of structure and associated semantics
  % Full definition of the intented data types by means of ADA declarations
  % may be appropiate
  }
  \ConsistencyRules{
  % Rules that the data in the structure is expected to obey in order
  % to be legal, even if not enforced by the program itself
  }

  \begin{code}
  --*/

#define MAX_NOF_T_WIDGETS 10

#define N_T_BTS 12
#define YR100  0
#define YR10   1
#define YR1    2
#define MT     3
#define DY10   4
#define DY1    5
#define HR10   6
#define HR1    7
#define MN10   8
#define MN1    9
#define SC10  10
#define SC1   11

struct time_widget {
  short usedp;           /* non-zero if entry is active */
  short first_component; /* TYPE = Time_Component */
  short last_component;  /* TYPE = Time_Component */
  struct tm timeval;     /* current value to be edited */
  wi_keycode tk;            /* wi_keycode of the text edit, 0 if not used */
  wi_keycode incs[N_T_BTS]; /* one for every time component */
  wi_keycode decs[N_T_BTS]; /* one for every time component */
  dwin dw;               /* where is it displayed */
  short x0;              /* origin x */
  short y0;              /* origin y */
  char editbuf[64];      /* current contents of the time buffer */
  char *resptr;          /* result pointer */
};

static struct time_widget tws[MAX_NOF_T_WIDGETS];
short tws_fill = 0;


/*--
  \end{code}

  \dataparagraph{Component specific information}

  \Definition{ % What the data structure is designed to represent
  }
  \SemanticStructure{
  % Description of structure and associated semantics
  % Full definition of the intented data types by means of ADA declarations
  % may be appropiate
  }
  \ConsistencyRules{
  % Rules that the data in the structure is expected to obey in order
  % to be legal, even if not enforced by the program itself
  }

  \begin{code}
  --*/

char *eng_month_names[13] = { "---",
  "Jan", "Feb", "Mar",
  "Apr", "May", "Jun",
  "Jul", "Aug", "Sep",
  "Oct", "Nov", "Dec" };

char *rom_month_names[13] = { "---",
  "Ian", "Feb", "Mar",
  "Apr", "Mai", "Iun",
  "Jul", "Aug", "Sep",
  "Oct", "Noe", "Dec" };

char **lang_month_names[6] = {
  eng_month_names, rom_month_names, NULL /* French */,
  NULL, NULL, NULL };

char **month_names = eng_month_names;

/*--
  \end{code}

  \funparagraph{clear\_time\_widget}
  % The name of the function is expected to be prezent
  % in the sectioning element above

  \PurposeReturn{
  Initializes a time widget, as an unused entry.
  }
  }
  \Preconditions{
  % Conditions to be met when the function is called
  }
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

void clear_time_widget(struct time_widget *tw)

{
  short i;
  
  tw->first_component = tw->last_component = 0; /* Inexistent Time_Component */
  tw->timeval.tm_sec = tw->timeval.tm_min = tw->timeval.tm_hour =
    tw->timeval.tm_mday = tw->timeval.tm_mon = tw->timeval.tm_year =
      tw->timeval.tm_wday = tw->timeval.tm_yday = tw->timeval.tm_isdst = 0;
  tw->tk = 0;
  tw->usedp = 0;
  for(i=0; i<N_T_BTS; i++)
    tw->incs[i] = tw->decs[i] = 0;
  return;
}


/*--
  \end{code}

  \funparagraph{asc\_time\_parse}
  % The name of the function is expected to be prezent
  % in the sectioning element above

  \PurposeReturn{
  Returns a time structure, containing the broken down
  values from the argument string, which must contain a time
  description similar to the ones produced by \functiontag{asctime}
  or \functiontag{ctime}.
  }
  \Exceptions{
  }
  \GlobalEffects{
  % Global effects on exported or external variables
  }
  \Preconditions{
  % Conditions to be met when the function is called
  }
  \Performance{
  % How it is, not why and what to do about it
  }
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

short asc_offs[SECOND];

static short init_asc_offs() {
  asc_offs[YEAR4] = 20;
  asc_offs[YEAR2] = 22;
  asc_offs[TEXT_MONTH] = 4;
  asc_offs[NUM_MONTH] = -1;
  asc_offs[DAY] = 8;
  asc_offs[HOUR] = 11;
  asc_offs[MINUTE] = 14;
  asc_offs[SECOND] = 17;
}

struct tm asc_time_parse(char *t)

{
  struct tm rt;  /* result time */
  short j;

  init_asc_offs();
  rt.tm_sec = atoi(t + asc_offs[SECOND]);
  rt.tm_min = atoi(t + asc_offs[MINUTE]);
  rt.tm_hour = atoi(t + asc_offs[HOUR]);
  rt.tm_mday = atoi(t + asc_offs[DAY]);
  
  rt.tm_mon = 0;
  for(j=1; j<13; j++) {
    if(strncmp(t+asc_offs[TEXT_MONTH], eng_month_names[j], 3)==0) {
      rt.tm_mon = j;
      break;
    }
  }
  rt.tm_year = atoi(t+asc_offs[YEAR2]);
  return rt;
}

/*--
  \end{code}

  \funparagraph{wi\_time\_panel}  % The name of the function(s)
  \ImplementationNotes{
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{
  This is one of the places which will need to be addapted to the
  multiple gwin version, when that version will be implemented.
  }

  \begin{code}
  --*/

char tw_buffer[64];  /* Contains the characters to edit */

static make_tw_buffer(struct tm tt, short from, short to)

{
  char *tb;
  short i;

  tb = tw_buffer;
  *tb++ = ' ';
  *tb = 0;
  for(i=from; i<=to; i++) {
    switch(i) {
    case YEAR2:
/*  case YEAR4:  DON'T use YEAR4 yet! */
      sprintf(tb, "%02d ", tt.tm_year);
      break;
    case TEXT_MONTH:
/*  case NUM_MONTH:  DON'T use NUM_MONTH yet! */
      sprintf(tb, "%3s ", month_names[tt.tm_mon]);
      break;
    case DAY:
      sprintf(tb, "%02d ", tt.tm_mday);
      break;
    case HOUR:
      sprintf(tb, "%02d:", tt.tm_hour);
      break;
    case MINUTE:
      sprintf(tb, "%02d:", tt.tm_min);
      break;
    case SECOND:
      sprintf(tb, "%02d", tt.tm_sec);
      break;
    default:
      break;
    }
    while(*tb!=0)
      tb++;
  }
  return;
}

short wi_time_panel(dwin d,               /* dwin in the current gwin */
		     pixed x0, pixed y0,   /* dwin position */
		     char *asct,           /* ASCII description of time */
		     short from, short to, /* Time_Components */
		     wi_keycode tkey,         /* wi_keycode of the time item */
		     wi_keycode inc0,         /* first increment wi_keycode */
		     wi_keycode dec0          /* first decrement wi_keycode */
		     )
{
  short i;
  short xoffs;
  short tws_ptr;
  short xch, ych;

  /*--
    \end{code}

    \subparagraph{Define the internal time representation}

    \begin{code}
    --*/

  for(tws_ptr=0; tws_ptr<tws_fill; tws_ptr++)
    if(tws[tws_ptr].usedp==0) break;
  if(tws_ptr==tws_fill)
    tws_fill ++;  /* Check here for the Too_Many_Time_Panels exception */
  clear_time_widget(tws + tws_ptr);
  tws[tws_ptr].first_component = from;
  tws[tws_ptr].last_component = to;
  tws[tws_ptr].timeval = asc_time_parse(asct);
  tws[tws_ptr].tk = tkey;
  tws[tws_ptr].x0 = x0;
  tws[tws_ptr].y0 = y0;
  tws[tws_ptr].dw = d;
  tws[tws_ptr].resptr = asct;
  tws[tws_ptr].usedp = 1;
  for(i=from; i<=to; i++) {
    switch(i) {
    case YEAR2:
      tws[tws_ptr].incs[YR10] = inc0++;
      tws[tws_ptr].decs[YR10] = dec0++;
      tws[tws_ptr].incs[YR1] = inc0++;
      tws[tws_ptr].decs[YR1] = dec0++;
      break;
    case TEXT_MONTH:
      tws[tws_ptr].incs[MT] = inc0++;
      tws[tws_ptr].decs[MT] = dec0++;
      break;
    case DAY:
      tws[tws_ptr].incs[DY10] = inc0++;
      tws[tws_ptr].decs[DY10] = dec0++;
      tws[tws_ptr].incs[DY1] = inc0++;
      tws[tws_ptr].decs[DY1] = dec0++;
      break;
    case HOUR:
      tws[tws_ptr].incs[HR10] = inc0++;
      tws[tws_ptr].decs[HR10] = dec0++;
      tws[tws_ptr].incs[HR1] = inc0++;
      tws[tws_ptr].decs[HR1] = dec0++;
      break;
    case MINUTE:
      tws[tws_ptr].incs[MN10] = inc0++;
      tws[tws_ptr].decs[MN10] = dec0++;
      tws[tws_ptr].incs[MN1] = inc0++;
      tws[tws_ptr].decs[MN1] = dec0++;
      break;
    case SECOND:
      tws[tws_ptr].incs[SC10] = inc0++;
      tws[tws_ptr].decs[SC10] = dec0++;
      tws[tws_ptr].incs[SC1] = inc0++;
      tws[tws_ptr].decs[SC1] = dec0++;
      break;
    default:
      break;
    } /* switch */
  } /* for */

  /*--
    \end{code}

    \subparagraph{Display time widget}

    \begin{code}
    --*/

  make_tw_buffer(tws[tws_ptr].timeval, from, to);
  strcpy(tws[tws_ptr].editbuf, tw_buffer);
  wi_textpad(d, x0, y0, strlen(tws[tws_ptr].editbuf), 64, 1,
	     tws[tws_ptr].editbuf, tkey);
  xoffs = 0;
  xch = wi_get_x_char_size();
  ych = wi_get_y_char_size();
  x0 += xch + 5;  /* To move everything to the right */
  for(i=from; i<=to; i++) {
    switch(i) {
    case YEAR2:
      wi_button(d, x0 + xoffs * xch - 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[YR10]);
      wi_button(d, x0 + xoffs * xch - 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[YR10]);
      xoffs ++;
      wi_button(d, x0 + xoffs * xch + 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[YR1]);
      wi_button(d, x0 + xoffs * xch + 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[YR1]);
      xoffs ++;
      xoffs ++;
      break;
    case TEXT_MONTH:
      wi_button(d, x0 + 5 + xoffs * xch, y0 - 8 - ych,
		xch * 3 - 10, ych, "", tws[tws_ptr].incs[MT]);
      wi_button(d, x0 + 5 + xoffs * xch, y0 - 6 + 2 * ych,
		xch * 3 - 10, ych, "", tws[tws_ptr].decs[MT]);
      xoffs +=4;
      break;
    case DAY:
      wi_button(d, x0 + xoffs * xch - 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[DY10]);
      wi_button(d, x0 + xoffs * xch - 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[DY10]);
      xoffs ++;
      wi_button(d, x0 + xoffs * xch + 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[DY1]);
      wi_button(d, x0 + xoffs * xch + 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[DY1]);
      xoffs ++;
      xoffs ++;
      break;
    case HOUR:
      wi_button(d, x0 + xoffs * xch - 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[HR10]);
      wi_button(d, x0 + xoffs * xch - 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[HR10]);
      xoffs ++;
      wi_button(d, x0 + xoffs * xch + 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[HR1]);
      wi_button(d, x0 + xoffs * xch + 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[HR1]);
      xoffs ++;
      xoffs ++;
      break;
    case MINUTE:
      wi_button(d, x0 + xoffs * xch - 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[MN10]);
      wi_button(d, x0 + xoffs * xch - 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[MN10]);
      xoffs ++;
      wi_button(d, x0 + xoffs * xch + 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[MN1]);
      wi_button(d, x0 + xoffs * xch + 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[MN1]);
      xoffs ++;
      xoffs ++;
      break;
    case SECOND:
      wi_button(d, x0 + xoffs * xch - 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[SC10]);
      wi_button(d, x0 + xoffs * xch - 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[SC10]);
      xoffs ++;
      wi_button(d, x0 + xoffs * xch + 2, y0 - 8 - ych,
		xch - 3, ych, "", tws[tws_ptr].incs[SC1]);
      wi_button(d, x0 + xoffs * xch + 2, y0 - 6 + 2 * ych,
		xch - 3, ych, "", tws[tws_ptr].decs[SC1]);
      xoffs ++;
      xoffs ++;
      break;
    default:
      break;
    } /* switch */
  } /* for */
} /* wi_time_panel */


/*--
  \end{code}

  \funparagraph{wi\_time\_del\_panel, wi\_time\_del\_dwin,
  wi\_time\_del\_all}
  \ImplementationNotes{
  % How implementation of the function is done: algorithm---if special,
  % implementation decisions, discarded implementation alternatives
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  }

  \begin{code}
  --*/

void wi_time_del_panel(wi_keycode key)

{
  short i;

  for(i=0; i<tws_fill; i++)
    if(tws[i].usedp!=0 && tws[i].tk==key) {
      tws[i].usedp = 0;
    }
}


void wi_time_del_dwin(dwin d)

{
  short i;

  for(i=0; i<tws_fill; i++)
    if(tws[i].usedp!=0 && tws[i].dw==d) {
      tws[i].usedp = 0;
    }
}


void wi_time_del_all()

{
  tws_fill = 0;
}


/*--
  \end{code}

  \funparagraph{wi\_time\_input}
  \ImplementationNotes{
  }
  \Portability{ % What is assumed about the operating environment
  }
  \Changes{ % that are anticipated, if special provision was made for them
  When the text in the time panel's textpad is changed, the
  previous text is copied in a security buffer. Then, the syntactical
  correctness of the newly entered text is checked, and --- if
  correct --- the old text is replaced and the time
  data adjusted, otherwise the old text is restored. This will
  be implemented, if prooven necessary, in the close future, because
  checking the date correctness would require potentially
  unjustified effort.
  }

  \begin{code}
  --*/

short month_len(short month, short year)

{
  switch(month) {
  case 1:
    return 31;
  case 2:
    if(year % 4 > 0)
      return 28;
    else
      return 29;
  case 3:
    return 31;
  case 4:
    return 30;
  case 5:
    return 31;
  case 6:
    return 30;
  case 7:
    return 31;
  case 8:
    return 31;
  case 9:
    return 30;
  case 10:
    return 31;
  case 11:
    return 30;
  case 12:
    return 31;
  default: /* should be an exception */
    return 30;
  } /* switch */
}


short wi_time_input(wi_keycode k)

{
  short i, j;

  for(i=0; i<tws_fill; i++) {
    if(tws[i].usedp==0)
      continue;
    if(tws[i].tk==k) { /* we have a textual buffer change */
      strcpy(tw_buffer, tws[i].editbuf);
      wi_accept();
      
      /* make the check and the parse here, but
	 continue if check fails */

      strcpy(tws[i].editbuf, tw_buffer); /* i.e. check failed */
      return 1;                          /* wi_keycode processed */
    }
    for(j=0; j<N_T_BTS; j++) {
      if(tws[i].incs[j]==k) {
	switch(j) {
	case YR10:
	  if(((tws[i].timeval.tm_year % 100) / 10) == 9)
	    tws[i].timeval.tm_year -= 90;
	  else
	    tws[i].timeval.tm_year += 10;
	  break;
	case YR1:
	  if((tws[i].timeval.tm_year % 10) == 9)
	    tws[i].timeval.tm_year -= 9;
	  else
	    tws[i].timeval.tm_year ++;
	  break;
	case MT:
	  if(tws[i].timeval.tm_mon==12)
	    tws[i].timeval.tm_mon = 1;
	  else
	    tws[i].timeval.tm_mon ++;
	  if(tws[i].timeval.tm_mday >
	     month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year))
	    tws[i].timeval.tm_mday =
	      month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year);
	  break;
	case DY10:
	  if((tws[i].timeval.tm_mday / 10)==3 ||
	     (tws[i].timeval.tm_mon==2 &&
	      (tws[i].timeval.tm_mday / 10)==2))
	    tws[i].timeval.tm_mday = tws[i].timeval.tm_mday % 10;
	  else
	    tws[i].timeval.tm_mday += 10;
	  if(tws[i].timeval.tm_mday >
	     month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year))
	    tws[i].timeval.tm_mday =
	      month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year);
	  break;
	case DY1:
	  if((tws[i].timeval.tm_mday % 10)==9 ||
	     (tws[i].timeval.tm_mon==2 && (tws[i].timeval.tm_year % 4 > 0)
	      && (tws[i].timeval.tm_mday % 10)==8))
	    tws[i].timeval.tm_mday = (tws[i].timeval.tm_mday / 10) * 10;
	  else
	    tws[i].timeval.tm_mday ++;
	  if(tws[i].timeval.tm_mday >
	     month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year))
	    tws[i].timeval.tm_mday =
	      month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year);
	  break;
	case HR10:
	  if(((tws[i].timeval.tm_hour % 100) / 10) == 2)
	    tws[i].timeval.tm_hour -= 20;
	  else
	    tws[i].timeval.tm_hour += 10;
	  break;
	case HR1:
	  if((tws[i].timeval.tm_hour % 10) == 9)
	    tws[i].timeval.tm_hour -= 9;
	  else
	    tws[i].timeval.tm_hour ++;
	  break;
	case MN10:
	  if(((tws[i].timeval.tm_min % 100) / 10) == 5)
	    tws[i].timeval.tm_min -= 50;
	  else
	    tws[i].timeval.tm_min += 10;
	  break;
	case MN1:
	  if((tws[i].timeval.tm_min % 10) == 9)
	    tws[i].timeval.tm_min -= 9;
	  else
	    tws[i].timeval.tm_min ++;
	  break;
	case SC10:
	  if(((tws[i].timeval.tm_sec % 100) / 10) == 5)
	    tws[i].timeval.tm_sec -= 50;
	  else
	    tws[i].timeval.tm_sec += 10;
	  break;
	case SC1:
	  if((tws[i].timeval.tm_sec % 10) == 9)
	    tws[i].timeval.tm_sec -= 9;
	  else
	    tws[i].timeval.tm_sec ++;
	  break;
	}
	make_tw_buffer(tws[i].timeval,
		       tws[i].first_component, tws[i].last_component);
	strcpy(tws[i].editbuf, tw_buffer);
	wi_inchg_text_val(tws[i].dw, tws[i].tk, tws[i].editbuf);
	return 1; /* Digit incrementation processed successfully */
      } /* if */
      if(tws[i].decs[j]==k) {
	switch(j) {
	case YR10:
	  if(((tws[i].timeval.tm_year % 100) / 10) == 0)
	    tws[i].timeval.tm_year += 90;
	  else
	    tws[i].timeval.tm_year -= 10;
	  break;
	case YR1:
	  if((tws[i].timeval.tm_year % 10) == 0)
	    tws[i].timeval.tm_year += 9;
	  else
	    tws[i].timeval.tm_year --;
	  break;
	case MT:
	  if(tws[i].timeval.tm_mon==1)
	    tws[i].timeval.tm_mon = 12;
	  else
	    tws[i].timeval.tm_mon --;
	  if(tws[i].timeval.tm_mday >
	     month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year))
	    tws[i].timeval.tm_mday =
	      month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year);
	  break;
	case DY10:
	  if((tws[i].timeval.tm_mday / 10) == 0)
	    tws[i].timeval.tm_mday = 30 + (tws[i].timeval.tm_mday % 10);
	      /* and will be adjusted below */
	  else
	    tws[i].timeval.tm_mday -= 10;
	  if(tws[i].timeval.tm_mday >
	     month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year))
	    tws[i].timeval.tm_mday =
	      month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year);
	  break;
	case DY1:
	  if(tws[i].timeval.tm_mday % 10 == 0)
	    tws[i].timeval.tm_mday += 9;
	  else
	    tws[i].timeval.tm_mday --;
	  if(tws[i].timeval.tm_mday >
	     month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year))
	    tws[i].timeval.tm_mday =
	      month_len(tws[i].timeval.tm_mon, tws[i].timeval.tm_year);
	  break;
	case HR10:
	  if(((tws[i].timeval.tm_hour % 100) / 10) == 0)
	    tws[i].timeval.tm_hour += 20;
	  else
	    tws[i].timeval.tm_hour -= 10;
	  break;
	case HR1:
	  if((tws[i].timeval.tm_hour % 10) == 0)
	    tws[i].timeval.tm_hour += 9;
	  else
	    tws[i].timeval.tm_hour --;
	  break;
	case MN10:
	  if(((tws[i].timeval.tm_min % 100) / 10) == 0)
	    tws[i].timeval.tm_min += 50;
	  else
	    tws[i].timeval.tm_min -= 10;
	  break;
	case MN1:
	  if((tws[i].timeval.tm_min % 10) == 0)
	    tws[i].timeval.tm_min += 9;
	  else
	    tws[i].timeval.tm_min --;
	  break;
	case SC10:
	  if(((tws[i].timeval.tm_sec % 100) / 10) == 0)
	    tws[i].timeval.tm_sec += 50;
	  else
	    tws[i].timeval.tm_sec -= 10;
	  break;
	case SC1:
	  if((tws[i].timeval.tm_sec % 10) == 0)
	    tws[i].timeval.tm_sec += 9;
	  else
	    tws[i].timeval.tm_sec --;
	  break;
	}
	make_tw_buffer(tws[i].timeval,
		       tws[i].first_component, tws[i].last_component);
	strcpy(tws[i].editbuf, tw_buffer);
	wi_inchg_text_val(tws[i].dw, tws[i].tk, tws[i].editbuf);
	return 1; /* Digit decrementation processed successfully */
      } /* if */
    } /* for(j ...) */
  } /* for(i ...) */
  return 0;
}


/*--
\end{code}

\funparagraph{wi\_time\_val}  % The name of the function(s)
\ImplementationNotes{ 
% How implementation of the function is done: algorithm---if special,
% implementation decisions, discarded implementation alternatives
}
\Portability{ % What is assumed about the operating environment
}
\Changes{ % that are anticipated, if special provision was made for them
}

\begin{code}
--*/

static char loctimeval[64] = "the loctimeval";
/* extern char *asctime(const struct tm *timeptr);
extern struct tm *gmtime(const time_t *timep);*/

char *wi_time_val(wi_keycode k)

{
  short i;
  long tt;

  for(i=0; i<tws_fill; i++) {
    if(tws[i].usedp==0)
      continue;
    if(tws[i].tk==k) {
      /* the following corrections are necessary because
	 the time value is represented by asc_time_parse
         and used by the various function in a way which is not
	 conformant to the Unix representation */
      tws[i].timeval.tm_mon --;
/*      tws[i].timeval.tm_hour ++; /* may have something to do
				    with timezone I believe */
      tt = mktime(&(tws[i].timeval));
      strncpy(loctimeval, ctime(&tt), 64);
      tws[i].timeval.tm_mon ++;
/*      tws[i].timeval.tm_hour --;*/
      return loctimeval;
    }
  }
  return NULL;
}

/*--
  \end{code}

  % MOVE THIS COMMENT TO THE END OF THE FILE
  \endoffile

  --*/

